
;;;;    Zusatzmodul SCHNEIDEREI fr AutoCAD R14 und 2000
;;;;    von Armin Antkowiak, Mrz 2000
;;;;    Funktionen NHEN, SCHNEIDEN, BGELN, POLYFIT

;;;;    Function set TAILORS for AutoCAD R14 and 2000
;;;;    by Armin Antkowiak, March 2000
;;;;    including functions SEW, XSLICE, IRON, POLYFIT



(if (not (or (wcmatch (ver) "*14*") (wcmatch (ver) "*2000*")))
   (alert
      (if (wcmatch (ver) "*(de)")
         (strcat
            "Diese Software wurde fr AutoCAD 14 und 2000 entwickelt.\n"
            "Da Sie eine andere Version benutzen, knnen Fehler auftreten."
         )
         (strcat
            "This software was developed for AutoCAD 14 and 2000.\n"
            "Errors may occur because you are using a different version."
         )
      )
   )
)



;____________________________________________________________________________;



(if (wcmatch (ver) "*(de)")



   ;;;   Funktion NHEN
   ;;;   fgt ein 3d-Polyflchennetz aus gewhlten Objekten zusammen.
   ;;;
   ;;;   Es knnen Punkte, Linien, 3d-Flchen,
   ;;;   Polygonnetze und Polyflchennetze zusammengefasst werden.
   ;;;   Diese mssen nicht notwendigerweise in rumlichem Zusammenhang
   ;;;   stehen [sie brauchen keine gemeinsamen Eckpunkte,
   ;;;   Kanten oder Flchen zu besitzen];
   ;;;   ein solcher Zusammenhang wird auch nicht hergestellt.
   ;;;   Alle Komponenten bleiben an ihrer Position.
   ;;;
   ;;;   Dem erzeugten Netz wird der Layer und die Farbe des
   ;;;   zuerst ausgewhlten Objekts zugewiesen
   ;;;   [analog zum AutoCAD-14-Befehl "Vereinig" fr Volumenkrper
   ;;;    und zum AutoCAD-Befehl "Pedit" fr Polylinien].
   ;;;
   ;;;   Es knnen nur Elemente ausgewhlt werden,
   ;;;   die keine Objekthhe besitzen.
   ;;;   Die Anzahl der Teilobjekte fr ein Polyflchennetz ist gewissen
   ;;;   Beschrnkungen unterworfen.
   ;;;
   ;;;   Ein Polyflchennetz kann mittels "Ursprung" in seine Bestandteile
   ;;;   zerlegt werden, d. h. in Punkte, Linien und 3d-Flchen.


   (defun c:nhen
      (
         /
         s       ; Auswahlsatz der zusammenzunhenden Objekte

         s#      ; Anzahl der ausgewhlten Objekte

         f#      ; Anzahl aller zusammenzufgenden Teilobjekte
                 ;    [Flchen, Linien, Punkte]

         i#      ; Index des aktuell bearbeiteten Objekts
         id      ; Elementdatenliste des aktuell bearbeiteten Objekts
         it      ; Typ des aktuell bearbeiteten Objekts

         i1 i2   ; Polygonnetz: M- und N-Wert
                 ;    [Anzahl der Scheitelpunkte pro Spalte bzw. Zeile]
                 ; Polyflchennetz: Anzahl der Scheitelpunkte und Teilobjekte

         v<      ; maximal zu erwartende Anzahl der Scheitelpunkte

         wm      ; warnende Botschaft
                 ;    [wird ausgegeben, wenn die Anzahl der Teilobjekte und
                 ;     Scheitelpunkte kritische Hhen erreicht]
         wn      ; identifizierende Nummer der geladenen Dialogfelddatei
         wr      ; Benutzerreaktion auf die Warnung
         wh      ; Name und vollstndiger Pfad der HTML-Hilfe-Datei

         tt      ; temporres Testflag

         ger     ; Flag: deutsche Version

         tol     ; Toleranz

         echo    ; Systemvariable "cmdecho" [command echo]
         errr    ; systemeigene Fehlerbearbeitungs-Routine
      )

      (setq ger t)
      (standardInitiate)
      (sewSelect)
      (sewProcess)
      (standardTerminate)
   )

)



;;;   Function SEW
;;;   creates a 3D polyface mesh composed of selected objects.
;;;
;;;   Points, lines, 3D faces, polygon meshes and polyface meshes
;;;   can be associated.
;;;   They don't necessarily have to be in spacial connection
;;;   [they do not need to share a common corner, edge, or area];
;;;   such a connention will not be generated by this function.
;;;   All components stay on their positions.
;;;
;;;   The first selected object determines layer and color
;;;   of the mesh created
;;;   [similar to AutoCAD 14 "union" command for 3D solids
;;;    and AutoCAD "pedit" command for polylines].
;;;
;;;   Objects with a non-zero thickness cannot be selected.
;;;   The number of components of a polyface mesh is limited.
;;;
;;;   The AutoCAD "explode" command dismantles a polyface mesh
;;;   into its components, i. e. points, lines, and 3D faces.


(defun c:sew
   (
      /
      s       ; selection set

      s#      ; number of objects selected

      f#      ; number of components to assemble [faces, lines, points]

      i#      ; index of object currently worked on
      id      ; entity data list
      it      ; type of entity

      i1 i2   ; polygon mesh: M and N value
              ;    [number of vertices per column and per row]
              ; polyface mesh: number of vertices and components

      v<      ; maximum expected number of vertices

      wm      ; warning message
              ;    [will be launched if number of vertices or components
              ;     reaches critical height]
      wn      ; identification number of dialog box file loaded
      wr      ; user's response to warning
      wh      ; name and path of HTML help file

      tt      ; temporary test flag

      ger     ; flag: German version

      tol     ; tolerance

      echo    ; "cmdecho" system variable [command echo]
      errr    ; system's error handling routine
   )

   (setq ger (wcmatch (ver) "*(de)"))
   (standardInitiate)
   (sewSelect)
   (sewProcess)
   (standardTerminate)
)



;;;   Unterprogramme 1. Ordnung fr NHEN
;;;   1st order subroutines for SEW


(defun sewSelect
   ( )

   (setq tt t)
   (while tt
      (princ
         (if ger
            " - Punkte, Linien, 3d-Flchen, Polygonnetze, Polyflchennetze -"
            " - points, lines, 3D faces, polygon meshes, polyface meshes -"
         )
      )
      (setq s
         (ssget
            '(
               (-4 . "<or")
                  (0 . "POINT")
                  (0 . "LINE")
                  (0 . "3DFACE")
                  (-4 . "<and")
                     (0 . "POLYLINE") (-4 . "&") (70 . 80)
                  (-4 . "and>")
               (-4 . "or>")
               (-4 . "=") (39 . 0.0)   ; zero thickness
            )
         )
      )
      (if s
         (setq
            s# (sslength s)
            tt nil
         )
         (princ
            (if ger
               "\nEs wurde keine gltige Auswahl getroffen."
               "\nNo valid selection made."
            )
         )
      )
   )
)



(defun sewProcess
   ( )

   (if (< 32767 s#)
      (princ
         (strcat "\n"
            (itoa s#)
            (if ger
               " Objekte knnen nicht zusammengefgt werden (maximal 32767)."
               " objects cannot be sewn together (not more than 32767)."
            )
         )
      )
      (progn
         (princ "\n")
         (setq
            f# 0
            v< 0
            i# 0
         )
         (while (> s# i#)
            (setq
               id (entget (ssname s i#))
               it (cdr (assoc 0 id))
               i# (1+ i#)
            )
            (cond
               (
                  (= "POLYLINE" it)   ; polygon mesh or polyface mesh
                  (if (zerop (logand 4 (setq it (cdr (assoc 70 id)))))
                     (setq
                        i1 (cdr (assoc 71 id))   ; not surface fit
                        i2 (cdr (assoc 72 id))
                     )
                     (setq
                        i1 (cdr (assoc 73 id))   ; surface fit
                        i2 (cdr (assoc 74 id))
                     )
                  )
                  (if (zerop (logand 64 it))
                     (setq                       ; polygon mesh
                        f#
                           (+
                              f#
                              (*
                                 (if (zerop (logand  1 it))
                                    (1- i1)      ; M open
                                    i1           ; M closed
                                 )
                                 (if (zerop (logand 32 it))
                                    (1- i2)      ; N open
                                    i2           ; N closed
                                 )
                              )
                           )
                        v< (+ v< (* i1 i2))
                     )
                     (setq                       ; polyface mesh
                        f# (+ f# i2)
                        v< (+ v< (min i1 (lsh i2 2)))
                     )   ; four vertices per face at most
                  )
               )
               (
                  (= "3DFACE" it)
                  (setq
                     f# (1+ f#)
                     v< (+ 4 v<)
                  )
               )
               (
                  (= "LINE" it)
                  (setq
                     f# (1+ f#)
                     v< (+ 2 v<)
                  )
               )
               (
                  t   ; (= "POINT" it)
                  (setq 
                     f# (1+ f#)
                     v< (1+ v<)
                  )
               )
            )
         )
         (cond
            (
               (< 32767 f#)   ; too many components
               (princ
                  (if ger
                     (strcat
                        "Ein Polyflchennetz mit "
                        (itoa f#)
                        " Teilen kann nicht erstellt werden (maximal 32767)."
                     )
                     (strcat
                        "Cannot assemble "
                        (itoa f#)
                        " components to a polyface mesh"
                        " (not more than 32767)."
                     )
                  )
               )
            )
            (
               (< 32767 v<)   ; possibly too many vertices
               (setq wm
                  (if ger
                     (strcat " "
                        (itoa f#)
                        " Teile sollen zusammengefgt werden.\n"
                        " Das kann mehrere Stunden dauern.\n"
                        " Es wird misslingen,"
                        " falls das entstehende Netz\n"
                        " mehr als 32767 verschiedene"
                        " Scheitelpunkte besitzt.\n\n"
                        " Soll es trotzdem versucht werden?"
                     )
                     (strcat " "
                        (itoa f#)
                        " components have to be assembled.\n"
                        " This process may take several hours.\n"
                        " It will fail if the arising mesh\n"
                        " has more than 32767 different vertices.\n\n"
                        " Start a try?"
                     )
                  )
               )
               (if
                  (and
                     (<
                        0
                        (setq wn
                           (load_dialog
                              (if ger
                                 "Tailors/Deutsch/Schneiderei.dcl"
                                 "Tailors/English/Tailors.dcl"
                              )
                           )
                        )
                     )
                     (new_dialog "warning" wn)
                  )
                  (progn   ; dialog box initiated successfully
                     (set_tile "message" wm)
                     (if
                        (setq wh
                           (findfile
                              (if ger
                                 "Tailors/Deutsch/Hilfe/netz.html"
                                 "Tailors/English/Help/mesh.html"
                              )
                           )
                        )
                        (action_tile "help" "(done_dialog 2)")
                        (mode_tile "help" 1)   ; help file not found
                     )
                     (action_tile "yes" "(done_dialog 1)")
                     (mode_tile (if (< 16383 f#) "no" "yes") 2)
                     (setq wr (start_dialog))
                     (unload_dialog wn)
                     (cond
                        (
                           (= 2 wr)   ; "help"
                           (command "_.browser" wh)
                        )
                        (
                           (= 1 wr)   ; "yes"
                           (sewProcessSet s f# v<)
                        )
                        (
                           t          ; "no"
                           nil
                        )
                     )
                  )
                  (progn   ; dialog box initiation failed
                     (initget (if ger "Ja Nein _Yes No" "Yes No"))
                     (textscr)
                     (terpri)
                     (princ wm)
                     (if (< 16383 f#)
                        (progn
                           (setq wr
                              (getkword
                                 (if ger "\n Ja/<Nein> : " "\n Yes/<No> : ")
                              )
                           )
                           (if (not wr) (setq wr  "No"))
                        )
                        (progn
                           (setq wr
                              (getkword
                                 (if ger "\n <Ja>/Nein : " "\n <Yes>/No : ")
                              )
                           )
                           (if (not wr) (setq wr "Yes"))
                        )
                     )
                     (terpri)
                     (graphscr)
                     (if (= "Yes" wr) (sewProcessSet s f# v<))
                  )
               )
            )
            (
               t   ; neither too many components nor too many vertices
               (sewProcessSet s f# v<)
            )
         )
      )
   )
)



;;;   Unterprogramm 2. Ordnung fr sewProcess
;;;   [wird auch von xsliceProcessMesh aufgerufen]

;;;   2nd order subroutine for sewProcess
;;;   [also called by xsliceProcessMesh]


(defun sewProcessSet
   (
      s       ; Auswahlsatz der zusammenzunhenden Objekte

      f#      ; Anzahl aller zusammenzufgenden Teilobjekte
              ;    [Flchen, Linien, Punkte]

      v<      ; maximal zu erwartende Anzahl der Scheitelpunkte
      /
      s#      ; Anzahl der Objekte im Auswahlsatz
      i#      ; Index des aktuell bearbeiteten Objekts
      in      ; Elementname des aktuell bearbeiteten Objekts
      id      ; Elementdatenliste des aktuell bearbeiteten Objekts
      it      ; Typ des aktuell bearbeiteten Objekts
      ie      ; Bitcode: Sichtbarkeit der Kanten

      i1 i2   ; fr Polygonnetz: M- und N-Wert
              ;    [Anzahl der Scheitelpunkte pro Spalte bzw. Zeile]
              ; fr Polyflchennetz:
              ;    Anzahl der Scheitelpunkte und Teilobjekte

      j1 j2   ; fr Polygonnetz: Index der aktuell bearbeiteten
              ;                     Zeile bzw. Spalte
              ; fr Polyflchennetz: Index des aktuell bearbeiteten
              ;                         Scheitelpunkts bzw. Teilobjekts
      
      mo no   ; Flags: Polygonnetz ist M-offen bzw. N-offen

      v^      ; Liste aller Scheitelpunkte
      v*      ; Teilliste der Scheitelpunkte, die noch nicht mit dem
              ;    aktuell bearbeiteten Punkt verglichen wurden
      v#      ; Anzahl der Scheitelpunkte
      v%      ; Anzahl der bisher bearbeiteten Punkte
      j#      ; Index des aktuell bearbeiteten Scheitelpunkts

      cc      ; aktuell bearbeiteter Punkt

      fr      ; Liste der Scheitelpunkt-Zuordnungen
              ;    fr das aktuell bearbeitete Objekt bzw. Teilobjekt
      ff      ; Liste der Scheitelpunkt-Zuordnungen
              ;    fr alle Zeilen eines Polygonnetzes
      f-      ; Liste der Scheitelpunkt-Zuordnungen fr die
              ;    erste der aktuell bearbeiteten Zeilen eines Polygonnetzes
      f=      ; Liste der Scheitelpunkt-Zuordnungen fr die
              ;    zweite der aktuell bearbeiteten Zeilen eines Polygonnetzes
      f1 f2   ; Scheitelpunkt-Zuordnungen fr die Eckpunkte
      f3 f4   ;    der aktuell bearbeiteten Teilflche eines Netzes

      f^      ; Liste der Scheitelpunkt-Zuordnungen fr alle Teilobjekte
              ;    [Flchen, Linien, Punkte]

      hc      ; Datengruppe:
              ;    Farbe des ersten ausgewhlten Teilobjekts
      hd      ; Datenliste:
              ;    Layer und Farbe des ersten ausgewhlten Teilobjekts
   )

   ;|
      s       ; selection set of objects to be sewn together

      f#      ; total number of components to be sewn together
              ;    [faces, lines, points]

      v<      ; maximum number of vertices anticipated
      /
      s#      ; number of objects in selection set
      i#      ; index of object currently worked on
      in      ; entity name
      id      ; entity data list
      it      ; type of entity
      ie      ; bit code: visibility of edges

      i1 i2   ; concerning a polygon mesh: M and N value
              ;    [number of vertices per column and per row]
              ; concerning a polyface mesh:
              ;    number of vertices and components

      j1 j2   ; concerning a polygon mesh:
              ;    index of current row and column
              ; concerning a polyface mesh:
              ;    index of current vertex and component
      
      mo no   ; flags: polygon mesh is open in M direction / N direction

      v^      ; list of all vertices
      v*      ; sublist of all vertices not compared with current point yet
      v#      ; total number of vertices
      v%      ; number of points compared already
      j#      ; index of current vertex

      cc      ; current point

      fr      ; face record [list of vertex assignments] for current component
      ff      ; list of face records for all rows of a polygon mesh
      f-      ; list of face records for the
              ;    first of the polygon mesh rows currently worked on
      f=      ; list of face records for the
              ;    second of the polygon mesh rows currently worked on
      f1 f2   ; vertex assignments for the corners
      f3 f4   ;    of a mesh component currently worked on

      f^      ; list of face records for all components [faces, lines, points]

      hc      ; data group: color of first selected component
      hd      ; data list: layer and color of first selected component
   |;


   ;;  Verarbeitung
   ;;  Processing

   (setq
      s# (sslength s)
      v# 0
      v% 0
      i# 0
   )
   (while (> s# i#)
      (setq
         in (ssname s i#)
         id (entget in)
         it (cdr (assoc 0 id))
         i# (1+ i#)
         fr nil
      )
      (cond
         (
            (= "POLYLINE" it)
            (if (zerop (logand 4 (setq it (cdr (assoc 70 id)))))
               (setq
                  i1 (cdr (assoc 71 id))   ; not surface fit
                  i2 (cdr (assoc 72 id))
               )
               (setq
                  i1 (cdr (assoc 73 id))   ; surface fit
                  i2 (cdr (assoc 74 id))
                  in (entnext in)   ; leap over first frame point entity
               )                    ; standing between header and
            )                       ; fit point entities
            (if (zerop (logand 64 it))
               (progn   ; polygon mesh
                  (setq
                     mo (zerop (logand  1 it))
                     no (zerop (logand 32 it))
                     ff nil
                     ie 0
                     j1 0
                  )
                  (while (> i1 j1)
                     (setq
                        j1 (1+ j1)
                        j2 0
                     )
                     (while (> i2 j2)
                        (setq
                           j2 (1+ j2)
                           in (entnext in)
                           id (entget in)
                        )
                        (foreach kn '(10) (sewProcessTestVertex))
                     )
                     (setq
                        ff (cons (if no fr (cons (last fr) fr)) ff)
                        fr nil
                     )
                  )
                  (if (not mo) (setq ff (cons (last ff) ff)))
                  (while
                     (setq
                        f- (car ff)
                        ff (cdr ff)
                        f= (car ff)
                     )
                     (while
                        (setq
                           f3 (car f-)
                           f2 (car f=)
                           f- (cdr f-)
                        )
                        (setq
                           f= (cdr f=)
                           f4 (car f-)
                           f1 (car f=)
                           f^ (cons (list f4 f3 f2 f1) f^)
                        )
                     )
                  )
               )
               (progn   ; polyface mesh
                  (setq
                     ie 0
                     j1 i1
                  )
                  (while (< 0 j1)
                     (setq
                        j1 (1- j1)
                        in (entnext in)
                        id (entget in)
                     )
                     (foreach kn '(10) (sewProcessTestVertex))
                  )
                  (while (> i2 j1)
                     (setq
                        j1 (1+ j1)
                        in (entnext in)
                        id (entget in)
                        f1 (cdr (assoc 71 id))
                        f2 (cdr (assoc 72 id))
                        f3 (cdr (assoc 73 id))
                        f4 (cdr (assoc 74 id))
                        f^
                           (cons
                              (list
                                 (cond
                                    (
                                       (zerop f4)
                                       0
                                    )
                                    (
                                       (minusp f4)
                                       (- (nth (+ i1 f4) fr))
                                    )
                                    (
                                       t
                                       (nth (- i1 f4) fr)
                                    )
                                 )
                                 (cond
                                    (
                                       (zerop f3)
                                       0
                                    )
                                    (
                                       (minusp f3)
                                       (- (nth (+ i1 f3) fr))
                                    )
                                    (
                                       t
                                       (nth (- i1 f3) fr)
                                    )
                                 )
                                 (cond
                                    (
                                       (zerop f2)
                                       0
                                    )
                                    (
                                       (minusp f2)
                                       (- (nth (+ i1 f2) fr))
                                    )
                                    (
                                       t
                                       (nth (- i1 f2) fr)
                                    )
                                 )
                                 (cond
                                    (
                                       (zerop f1)
                                       0
                                    )
                                    (
                                       (minusp f1)
                                       (- (nth (+ i1 f1) fr))
                                    )
                                    (
                                       t
                                       (nth (- i1 f1) fr)
                                    )
                                 )
                              )
                              f^
                           )
                     )
                  )
               )
            )
         )
         (
            (= "3DFACE" it)
            (setq ie (cdr (assoc 70 id)))
            (foreach kn '(10 11 12 13) (sewProcessTestVertex))
            (setq f^ (cons fr f^))
         )
         (
            (= "LINE" it)
            (setq ie 0)
            (foreach kn '(10 11) (sewProcessTestVertex))
            (setq f^ (cons (cons 0 (cons 0 fr)) f^))
         )
         (
            t   ; (= "POINT" it)
            (setq ie 0)
            (foreach kn '(10) (sewProcessTestVertex))
            (setq f^ (cons (cons 0 (cons 0 (cons 0 fr))) f^))
         )
      )
   )


   ;;  Ausgabe
   ;;  Output

   (if (< 32767 v#)
      (progn
         (princ
            (if ger
               (strcat "\015"
                  " Ein Polyflchennetz mit "
                  (itoa v#)
                  " Scheitelpunkten kann nicht erstellt werden"
                  " (maximal 32767)."
               )
               (strcat "\015"
                  " Cannot create a polyface mesh with "
                  (itoa v#)
                  " vertices (not more than 32767)."
               )
            )
         )
      )
      (progn                               ; layer and color of mesh
         (setq id (entget (ssname s 0)))   ; should correspond to
         (entmake                          ; first selected object
            (append
               '((0 . "POLYLINE"))
               (setq hd
                  (cons (assoc 8 id) (if (setq hc (assoc 62 id)) (list hc)))
               )
               (list
                  '(66 . 1)           ; "vertex entities follow" flag
                  '(10 0.0 0.0 0.0)   ; "dummy" point
                  '(70 . 64)          ; "polyface mesh"
                  (cons 71 v#)        ; number of vertices
                  (cons 72 f#)        ; number of faces
               )
            )
         )
         (setq j# v#)
         (while (< 0 j#)
            (setq j# (1- j#))
            (entmake
               (append
                  '((0 . "VERTEX"))
                  hd                         ; layer and color
                  (list
                     (cons 10 (nth j# v^))   ; vertex coordinates
                     '(70 . 192)             ; "polyface mesh vertex"
                  )
               )
            )
         )
         (while
            (setq fr (car f^))
            (setq f^ (cdr f^))
            (entmake
               (append
                  '((0 . "VERTEX"))
                  hd                         ; layer and color
                  (list
                     '(10 0.0 0.0 0.0)       ; "dummy" point
                     '(70 . 128)             ; "face record"
                     (cons 71 (cadddr fr))
                     (cons 72 (caddr  fr))
                     (cons 73 (cadr   fr))   ; vertex assignments
                     (cons 74 (car    fr))   ;    for corners of face
                  )
               )
            )
         )
         (entmake
            (append
               '((0 . "SEQEND"))     ; end of sequence
               hd                    ; layer and color
            )
         )
         (command "_.erase" s "")
         (if (< 62 v%) (princ "\015                        \015"))
      )
   )
)



;;;   Unterprogramm 3. Ordnung fr sewProcessSet
;;;   3rd order subroutine for sewProcessSet


(defun sewProcessTestVertex
   ( )

   (if (= 63 (logand 63 v%))
      (princ
         (if ger
            (strcat "\015"
               "Netz zu "
               (itoa (fix (/ (* 100.0 v% v%) v< v<)))
               "% fertig"
            )
            (strcat "\015"
               "mesh completed to "
               (itoa (fix (/ (* 100.0 v% v%) v< v<)))
               "%"
            )
         )
      )   ; processing time is proportional to squared number of vertices
   )
   (setq
      cc (cdr (assoc kn id))
      v% (1+ v%)
      v* v^
      j# v#
   )
   (while (< 0 j#)
      (if (equal cc (car v*) tol)   ; If point is arleady contained
         (setq                      ; by vertex list,
            fr
               (cons
                  (if (zerop (logand (lsh 1 (- kn 10)) ie))
                     j#
                     (- j#)
                  )                 ; then store number of vertex
                  fr                ; in face record;
               )
            j# -1
         )
         (setq
            j# (1- j#)
            v* (cdr v*)
         )
      )
   )
   (if (= 0 j#)
      (setq
         v^ (cons cc v^)            ; otherwise add point
         v# (1+ v#)                 ; to vertex list
         fr                         ; and store number in face record.
            (cons
               (if (zerop (logand (lsh 1 (- kn 10)) ie))
                  v#
                  (- v#)
               )
               fr
            )
      )
   )
)



;____________________________________________________________________________;



(if (wcmatch (ver) "*(de)")



   ;;;   Funktion SCHNEIDEN
   ;;;   kappt Linien, Strahlen, Konstruktionslinien,
   ;;;   3d-Flchen, Polygonnetze und Polyflchennetze an einer Ebene.
   ;;;
   ;;;   Die Optionen entsprechen denen des AutoCAD-Befehls "Kappen"
   ;;;   fr Volumenkrper.
   ;;;
   ;;;   Wenn es Objekte gibt, die zwar nicht von der Kappebene geschnitten
   ;;;   werden, aber auf der unerwnschten Seite liegen,
   ;;;   dann wird das Lschen dieser Objekte angeboten.
   ;;;
   ;;;   Objekte mit einer von Null verschiedenen Objekthhe
   ;;;   knnen nicht ausgewhlt werden.
   ;;;   Objekte auf gesperrten Layern werden nicht geschnitten.
   ;;;
   ;;;   Die geschnittenen Bestandteile eines Netzes
   ;;;   diesseits bzw. jenseits der Kappebene werden zu jeweils
   ;;;   einem Polyflchennetz zusammengefgt,
   ;;;   sofern ihre Anzahl [pro Seite] nicht 8191 bersteigt.
   ;;;   Andernfalls bleiben die Flchen, Linien bzw. Punkte
   ;;;   als einzelne Objekte bestehen.


   (defun c:schneiden
      (
         /
         s          ; Auswahlsatz der zu kappenden Objekte
         u          ; Satz der ausgewhlten Objekte,
                    ;      die gnzlich auf der unerwnschten Seite liegen

         p1 p2 p3   ; Punkte, welche die Kappebene definieren
         nv         ; Normalenvektor der Kappebene
         en         ; Elementname des ausgewhlten schneidenden Objekts
         ed         ; Elementdatenliste des schneidenden Objekts
         et         ; Typ des schneidenden Objekts
         eh         ; Objekthhe des schneidenden Objekts
         h*         ; Nummer des Ansichtsfensters, das whrend des Markierens
                    ;        des schneidenden Objekts aktuell ist
         h~         ; Nummer des Ansichtsfensters, das bei der Seitenwahl
                    ;        aktuell ist

         d+ d-      ; Flags: Seite mit positivem bzw. negativem Abstand
                    ;        von der Kappebene ist erwnscht

         s#         ; Anzahl der Objekte [auf nicht gesperrten Layern]
         l#         ; Anzahl der Objekte auf gesperrten Layern
         n#         ; Anzahl der von der Kappebene nicht geschnittenen Objekte
         u#         ; Anzahl der ganz auf der
                    ;        unerwnschten Seite liegenden Objekte
         i#         ; Index des aktuell bearbeiteten Objekts
         in         ; Elementname des aktuell bearbeiteten Objekts
         id         ; Elementdatenliste des aktuell bearbeiteten Objekts
         it         ; Typ des aktuell bearbeiteten Objekts
         ie         ; Datengruppe: Sichtbarkeit der Kanten

         i0 i1      ; Datengruppen, die die Punkte bzw. Richtungsvektoren
         i2 i3      ;               des aktuell bearbeiteten Objekts enthalten

         c0 c1      ; Eckpunkte der aktuell bearbeiteten 3d-Flche
         c2 c3

         d0 d1      ; Abstnde der Punkte von der Kappebene
         d2 d3

         d< d>      ; Maximum bzw. Minimum der Abstnde von der Kappebene

         ip iq      ; Schnittpunkte mit der Kappebene
         ir is

         b          ; Auswahlsatz: Teilobjekte aus Zerlegung eines Netzes
         b+ b-      ; davon auf der positiven bzw. negativen Seite
         b*         ; Systemvariable "splframe" vor dem Zerlegen des Netzes
         b#         ; Anzahl der durch Zerlegung entstandenen Teilobjekte
         j#         ; Index des aktuell bearbeiteten Teilobjekts
         jn         ; Elementname des aktuell bearbeiteten Scheitelpunkts
         jd         ; Elementdatenliste des Scheitelpunkts
         jt         ; Datengruppe: Vertex Flags
         v+ v-      ; maximal zu erwartende Anzahl der Scheitelpunkte
                    ;         auf der positiven bzw. negativen Seite

         ld         ; Datenliste des aktuell berprften Layers
         ll         ; Liste aller gesperrten Layer der Zeichnung

         tt         ; temporres Testflag

         r14        ; Flag: Release 14
         ger        ; Flag: deutsche Version

         tol        ; Toleranz

         echo       ; Systemvariable "cmdecho" [command echo]
         errr       ; systemeigene Fehlerbearbeitungs-Routine
      )

      (setq ger t)
      (xsliceInitiate)
      (xsliceSelect)
      (xsliceInput)
      (lockedFilter)
      (xsliceProcess)
      (standardTerminate)
   )

)



;;;   Function XSLICE
;;;   slices lines, rays, xlines,
;;;   3D faces, polygon meshes, and polyface meshes with a plane.
;;;
;;;   Options of XSLICE are similar to those of the AutoCAD "slice"
;;;   command for 3D solids.
;;;
;;;   If there are objects not intersected by the slicing plane
;;;   but situated on the undesired side of the plane,
;;;   XSLICE will offer to erase these objects.
;;;
;;;   Objects with a non-zero thickness cannot be selected.
;;;   Objects on locked layers do not get sliced.
;;;
;;;   All the sliced mesh components on the desired side of the plane
;;;   are reassembled to a polyface mesh
;;;   if their total number does not exceed 8191 [per side].
;;;   Otherwise the faces, lines, and points
;;;   will remain individual objects.


(defun c:xslice
   (
      /
      s          ; selection set of objects to be sliced
      u          ; set of selected objects situated entirely
                 ;     on the undesired side of the slicing plane

      p1 p2 p3   ; points defining the slicing plane
      nv         ; normal vector of the slicing plane
      en         ; entity name of object selected to define slicing plane
      ed         ; entity data list of slicing object
      et         ; type of slicing object
      eh         ; thickness of slicing object
      h*         ; number of viewport current when highlighting
                 ;        slicing object
      h~         ; number of viewport current when choosing
                 ;        desired side[s]

      d+ d-      ; flags: side with positive or negative distance
                 ;        from slicing plane is desired

      s#         ; number of objects [on unlocked layers]
      l#         ; number of objects on locked layers
      n#         ; number of objects not intersected by slicing plane
      u#         ; number of objects situated entirely
                 ;     on the undesired side of the slicing plane
      i#         ; index of object currently worked on
      in         ; entity name
      id         ; entity data list
      it         ; type of entity
      ie         ; data group: visibility of edges

      i0 i1      ; data groups containing points or direction vectors
      i2 i3      ;             of object currently worked on

      c0 c1      ; corners of 3D face currently worked on
      c2 c3

      d0 d1      ; distance of points from slicing plane
      d2 d3

      d< d>      ; maximum and minimum distances from slicing plane

      ip iq      ; intersection points with slicing plane
      ir is

      b          ; selection set: components of a dismantled mesh
      b+ b-      ; set of components on positive and negative side
      b*         ; "splframe" system variable before dismantling mesh
      b#         ; number of components got by dismantling
      j#         ; index of component currently worked on
      jn         ; entity name of current vertex
      jd         ; entity data list of current vertex
      jt         ; data group: vertex flags
      v+ v-      ; maximum number of vertices anticipated
                 ;         on positive and negative side

      ld         ; data list of layer currently tested
      ll         ; list of all locked layers of the drawing

      tt         ; temporary test flag

      r14        ; flag: release 14
      ger        ; flag: German version

      tol        ; tolerance

      echo       ; "cmdecho" system variable [command echo]
      errr       ; system's error handling routine
   )

   (setq ger (wcmatch (ver) "*(de)"))
   (xsliceInitiate)
   (xsliceSelect)
   (xsliceInput)
   (lockedFilter)
   (xsliceProcess)
   (standardTerminate)
)



;;;   Unterprogramme 1. Ordnung fr SCHNEIDEN
;;;   1st order subroutines for XSLICE


(defun xsliceSelect
   ( )

   (setq tt t)
   (while tt
      (princ
         (if ger
            (strcat
               " - Linien, Strahlen, Konstruktionslinien,"
               " 3d-Flchen, Polygonnetze, Polyflchennetze -"
            )
            (strcat
               " - lines, rays, xlines,"
               " 3D faces, polygon meshes, polyface meshes -"
            )
         )
      )
      (setq s
         (ssget
            '(
               (-4 . "<or")
                  (0 . "LINE")
                  (0 . "RAY")
                  (0 . "XLINE")
                  (0 . "3DFACE")
                  (-4 . "<and")
                     (0 . "POLYLINE") (-4 . "&") (70 . 80)
                  (-4 . "and>")
               (-4 . "or>")
               (-4 . "=") (39 . 0.0)   ; zero thickness
            )
         )
      )
      (if s
         (setq
            s# (sslength s)
            tt nil
         )
         (princ
            (if ger
               "\nEs wurde keine gltige Auswahl getroffen."
               "\nNo valid selection made."
            )
         )
      )
   )
)



(defun xsliceInput
   ( )

   (setq r14 (wcmatch (ver) "*14*"))


   ;;  Kappebene whlen
   ;;  Define slicing plane

   (initget
      (if ger
         (strcat
            "Objekt ZAchse Ansicht XY YZ ZX 3Punkte"
            " _Object Zaxis View XY YZ ZX 3points"
         )
         "Object Zaxis View XY YZ ZX 3points"
      )
   )
   (setq p1
      (getpoint
         (if ger
            (if r14
               "\nKappebene von Objekt/ZAchse/Ansicht/XY/YZ/ZX/<3Punkte>: "
               (strcat "\n"
                  "Ersten Punkt auf der Kappebene angeben oder "
                  "[Objekt/ZAchse/Ansicht/XY/YZ/ZX/3Punkte] <3Punkte>: "
               )
            )
            (if r14
               "\nSlicing plane by Object/Zaxis/View/XY/YZ/ZX/<3points>: "
               (strcat "\n"
                  "Specify first point on slicing plane by "
                  "[Object/Zaxis/View/XY/YZ/ZX/3points] <3points>: "
               )
            )
         )
      )
   )
   (if (= "Object" p1)
      (progn
         (setq tt t)
         (while tt
            (setq en
               (car
                  (entsel
                     (if ger
                        (strcat "\n"
                           "Zweidimensionales Objekt whlen,"
                           " das die Kappebene definiert: "
                        )
                        (strcat "\n"
                           "Select a two-dimensional object"
                           " defining slicing plane: "
                        )
                     )
                  )
               )
            )
            (if en
               (progn
                  (setq
                     ed (entget en)
                     et (cdr (assoc 0 ed))
                  )
                  (if
                     (and
                        (or
                           (= "CIRCLE" et)
                           (= "ARC" et)
                           (= "ELLIPSE" et)
                           (= "LWPOLYLINE" et)
                           (and
                              (= "POLYLINE" et)
                              (= 0 (logand 88 (cdr (assoc 70 ed))))
                           )   ; no 3D polylinie, no mesh
                           (and
                              (= "SPLINE" et)
                              (= 8 (logand 24 (cdr (assoc 70 ed))))
                           )   ; planar non-linear spline only
                        )
                        (if (setq eh (cdr (assoc 39 ed)))
                           (= 0.0 eh)   ; zero thickness
                           t            ; if group 39 exists
                        )
                     )
                     (setq tt nil)  ; selection succeeded
                     (princ
                        (if ger
                           (strcat "\n"
                              "Ungltige Auswahl;"
                              " Ebene kann nicht extrahiert werden."
                           )
                           (strcat "\n"
                              "Unable to extract the plane"
                              " of the selected object."
                           )
                        )
                     )
                  )
               )
               (princ
                  (if ger
                     "\nEs wurde nichts ausgewhlt."
                     "\nNothing selected."
                  )
               )
            )
         )
         (redraw en 3)   ; highlight selected slicing object
         (setq
            h* (getvar "cvport")
            nv (cdr (assoc 210 ed))   ; extrusion direction in WCS
            p1
               (trans
                  (if (= "LWPOLYLINE" et)
                     (list
                        (cadr  (assoc 10 ed))   ; first vertex
                        (caddr (assoc 10 ed))   ; as 2D point in OCS;
                        (cdr   (assoc 38 ed))   ; elevation in
                     )                          ; OCS Z direction
                     (cdr (assoc 10 ed))   ; other types: center,
                  )                        ; first control point or "dummy"
                  en                       ; as 3D point in OCS
                  0   ; translate from OCS into WCS
               )
         )
      )
      (progn
         (cond
            (
               (= "Zaxis" p1)
               (initget 1)   ; not just "Enter"
               (setq
                  p1
                     (trans
                        (getpoint
                           (if ger
                              (if r14
                                 "\nPunkt auf der Kappebene: "
                                 "\nPunkt auf der Kappebene angeben: "
                              )
                              (if r14
                                 "\nPoint on slicing plane: "
                                 "\nSpecify a point on the slicing plane: "
                              )
                           )
                        )
                        1
                        0
                     )
                  tt t
               )
               (while tt
                  (initget 1)   ; not just "Enter"
                  (setq
                     p2
                        (trans
                           (getpoint
                              (trans p1 0 1)
                              (if ger
                                 (if r14
                                    (strcat "\n"
                                       "Punkt auf der z-Achse "
                                       "(Normale zur Kappebene): "
                                    )
                                    (strcat "\n"
                                       "Punkt auf der z-Achse "
                                       "(Normale zur Kappebene) angeben: "
                                    )
                                 )
                                 (if r14
                                    (strcat "\n"
                                       "Point on Z-axis "
                                       "(normal) of the plane: "
                                    )
                                    (strcat "\n"
                                       "Specify a point on the Z-axis "
                                       "(normal) of the plane: "
                                    )
                                 )
                              )
                           )
                           1
                           0
                        )
                     nv (normalize (mapcar '- p2 p1))
                  )
                  (if nv
                     (setq tt nil)   ; selection succeeded
                     (princ
                        (if ger
                           "\nDie Punkte drfen nicht identisch sein."
                           "\nThe two points must not be identical."
                        )
                     )
                  )
               )
            )
            (
               (= "View" p1)
               (setq
                  nv (trans (normalize (getvar "viewdir")) 1 0 t)
                  p1
                     (getpoint
                        (if ger
                           (if r14
                              "\nPunkt auf der Ansichtsebene <0,0,0>: "
                              (strcat "\n"
                                 "\nPunkt auf der aktuellen "
                                 "Ansichtsebene angeben <0,0,0>: "
                              )
                           )
                           (if r14
                              "\nPoint on view plane <0,0,0>: "
                              (strcat "\n"
                                 "Specify a point on the "
                                 "current view plane <0,0,0>: "
                              )
                           )
                        )
                     )
                  p1 (trans (if p1 p1 '(0.0 0.0 0.0)) 1 0)
               )
            )
            (
               (= "XY" p1)
               (setq
                  nv (trans '(0.0 0.0 1.0) 1 0 t)
                  p1
                     (getpoint
                        (if ger
                           (if r14
                              (strcat "\n"
                                 "Punkt auf der zur xy-Ebene "
                                 "parallelen Kappebene <0,0,0>: "
                              )
                              (strcat "\n"
                                 "Punkt auf der zur xy-Ebene parallelen "
                                 "Kappebene angeben <0,0,0>: "
                              )
                           )
                           (if r14
                              (strcat "\n"
                                 "Point on slicing plane the latter being "
                                 "parallel to XY plane <0,0,0>: "
                              )
                              (strcat "\n"
                                 "Specify a point on the slicing plane "
                                 "the latter being parallel "
                                 "to the XY-plane <0,0,0>: "
                              )
                           )
                        )
                     )
                  p1 (trans (if p1 p1 '(0.0 0.0 0.0)) 1 0)
               )
            )
            (
               (= "YZ" p1)
               (setq
                  nv (trans '(1.0 0.0 0.0) 1 0 t)
                  p1
                     (getpoint
                        (if ger
                           (if r14
                              (strcat "\n"
                                 "Punkt auf der zur yz-Ebene "
                                 "parallelen Kappebene <0,0,0>: "
                              )
                              (strcat "\n"
                                 "Punkt auf der zur yz-Ebene parallelen "
                                 "Kappebene angeben <0,0,0>: "
                              )
                           )
                           (if r14
                              (strcat "\n"
                                 "Point on slicing plane the latter being "
                                 "parallel to YZ plane <0,0,0>: "
                              )
                              (strcat "\n"
                                 "Specify a point on the slicing plane "
                                 "the latter being parallel "
                                 "to the YZ-plane <0,0,0>: "
                              )
                           )
                        )
                     )
                  p1 (trans (if p1 p1 '(0.0 0.0 0.0)) 1 0)
               )
            )
            (
               (= "ZX" p1)
               (setq
                  nv (trans '(0.0 1.0 0.0) 1 0 t)
                  p1
                     (getpoint
                        (if ger
                           (if r14
                              (strcat "\n"
                                 "Punkt auf der zur zx-Ebene "
                                 "parallelen Kappebene <0,0,0>: "
                              )
                              (strcat "\n"
                                 "Punkt auf der zur zx-Ebene parallelen "
                                 "Kappebene angeben <0,0,0>: "
                              )
                           )
                           (if r14
                              (strcat "\n"
                                 "Point on slicing plane the latter being "
                                 "parallel to ZX plane <0,0,0>: "
                              )
                              (strcat "\n"
                                 "Specify a point on the slicing plane "
                                 "the latter being parallel "
                                 "to the ZX-plane <0,0,0>: "
                              )
                           )
                        )
                     )
                  p1 (trans (if p1 p1 '(0.0 0.0 0.0)) 1 0)
               )
            )
            (
               t   ; option "3points"
               (if (/= 'list (type p1))   ; in case the first point
                  (progn                  ; was not clicked on yet
                     (initget 1)
                     (setq p1
                        (trans
                           (getpoint
                              (if ger
                                 (if r14
                                    "\nErster Punkt der Kappebene: "
                                    "\nErsten Punkt der Kappebene angeben: "
                                 )
                                 (if r14
                                    "\nFirst point on slicing plane: "
                                    "\nSpecify first point on slicing plane: "
                                 )
                              )
                           )
                           1
                           0
                        )
                     )
                  )
                  (setq p1 (trans p1 1 0))
               )
               (setq tt t)
               (while tt
                  (initget 1)   ; not just "Enter"
                  (setq p2
                     (trans
                        (getpoint
                           (trans p1 0 1)
                           (if ger
                              (if r14
                                 "\nZweiter Punkt der Kappebene: "
                                 "\nZweiten Punkt der Kappebene angeben: "
                              )
                              (if r14
                                 "\nSecond point on slicing plane: "
                                 "\nSpecify second point on slicing plane: "
                              )
                           )
                        )
                        1
                        0
                     )
                  )
                  (if (equal p1 p2 tol)
                     (princ
                        (if ger
                           "\nDie Punkte drfen nicht identisch sein."
                           "\nThe points must not be identical."
                        )
                     )
                     (setq tt nil)   ; selection succeeded
                  )
               )
               (setq tt t)
               (while tt
                  (initget 1)   ; not just "Enter"
                  (setq
                     p3
                        (trans
                           (getpoint
                              (trans p2 0 1)
                              (if ger
                                 (if r14
                                    "\nDritter Punkt der Kappebene: "
                                    "\nDritten Punkt der Kappebene angeben: "
                                 )
                                 (if r14
                                    "\nThird point on slicing plane: "
                                    "\nSpecify third point on slicing plane: "
                                 )
                              )
                           )
                           1
                           0
                        )
                     nv
                        (normalize
                           (vectorProduct (mapcar '- p2 p1) (mapcar '- p3 p1))
                        )
                  )
                  (if nv
                     (setq tt nil)   ; selection succeeded
                     (princ
                        (if ger
                           "\nDie Punkte drfen nicht kollinear sein."
                           "\nThe points must not be collinear."
                        )
                     )
                  )
               )
            )
         )
      )
   )


   ;;  Gewnschte Seite[n] auswhlen
   ;;  Choose desired side[s]

   (setq tt t)
   (while tt
      (initget 1 (if ger "Beide _Both" "Both"))
      (setq d+
         (getpoint
            (if ger
               (if r14
                  (strcat "\n"
                     "Beide seiten/"
                     "<Punkt auf der gewnschten Seite der Kappebene>: "
                  )
                  (strcat "\n"
                     "Punkt auf der gewnschten Seite der Kappebene angeben "
                     "oder [Beide]: "
                  )
               )
               (if r14
                  (strcat "\n"
                     "Both sides/"
                     "<point on desired side of the plane>: "
                  )
                  (strcat "\n"
                     "Specify a point on desired side of the plane "
                     "or [keep Both sides]: "
                  )
               )
            )
         )
      )
      (if (= "Both" d+)
         (setq
            d+ t
            d- t
            tt nil
         )
         (progn
            (setq d+
               (scalarProduct nv (mapcar '- (trans d+ 1 0) p1))
            )   ; distance between specified point and slicing plane
            (if (equal 0.0 d+ tol)
               (princ
                  (if ger
                     "\nDer Punkt darf sich nicht auf der Kappebene befinden."
                     "\nThe point must not be on the slicing plane."
                  )
               )
               (if (minusp d+)
                  (setq
                     d+ nil   ; side of negative distances is desired,
                     d- t     ; i. e. the side nv does not point at
                     tt nil
                  )
                  (setq
                     d+ t     ; side of positive distances is desired,
                     d- nil   ; i. e. the side nv points at
                     tt nil
                  )
               )
            )
         )
      )
   )
   (if h*
      (progn
         (setq h~ (getvar "cvport"))
         (setvar "cvport" h*)
         (redraw en 4)          ; unhighlight slicing object if required
         (setvar "cvport" h~)
      )
   )
)



(defun xsliceProcess
   ( )

   (if s
      (progn
         (princ "\n")
         (setq
            u  (ssadd)
            u# 0
            n# 0
            i# 0
         )
         (while (> s# i#)
            (setq
               in (ssname s i#)
               id (entget in)
               it (cdr (assoc 0 id))
            )
            (cond
               (
                  (= "LINE" it)
                  (setq
                     i0 (assoc 10 id)
                     i1 (assoc 11 id)
                     ; data groups of start point and end point
                     d0 (scalarProduct nv (mapcar '- (cdr i0) p1))
                     d1 (scalarProduct nv (mapcar '- (cdr i1) p1))
                     ; distances from slicing plane
                  )
                  (if
                     (or
                        (and (<=    tol  d0) (>= (- tol) d1))
                        (and (>= (- tol) d0) (<=    tol  d1))
                     )
                     ; slice only if points are on different sides
                     (xsliceProcessLine)
                     (doNotSlice)
                  )
               )
               (
                  (= "RAY" it)
                  (setq
                     i0 (assoc 10 id)
                     i1 (assoc 11 id)
                     ; data groups of start point and direction vector
                     d0 (scalarProduct nv (mapcar '- (cdr i0) p1))
                     d1 (scalarProduct nv (cdr i1))
                     ; distance and direction coponent normal to slicing plane
                  )
                  (if
                     (or
                        (and (<=    tol  d0) (>= (- tol) d1))
                        (and (>= (- tol) d1) (<=    tol  d1))
                     )
                     ; slice only if direction vector points at the side
                     ; where the start point is not situated on
                     (xsliceProcessRay)
                     (doNotSlice)
                  )
               )
               (
                  (= "XLINE" it)
                  (setq
                     i0 (assoc 10 id)
                     i1 (assoc 11 id)
                     ; data groups of "center" point and direction vector
                     d0 (scalarProduct nv (mapcar '- (cdr i0) p1))
                     d1 (scalarProduct nv (cdr i1))
                     ; distance and direction coponent normal to slicing plane
                  )
                  (if
                     (equal 0.0 d1 tol)
                     ; slice only if not parallel to slicing plane
                     (doNotSlice)
                     (xsliceProcessXline)
                  )
               )
               (
                  (= "3DFACE" it)
                  (setq
                     c0 (cdr (setq i0 (assoc 10 id)))
                     c1 (cdr (setq i1 (assoc 11 id)))
                     c2 (cdr (setq i2 (assoc 12 id)))
                     c3 (cdr (setq i3 (assoc 13 id)))
                     ; corners
                     d0 (scalarProduct nv (mapcar '- c0 p1))
                     d1 (scalarProduct nv (mapcar '- c1 p1))
                     d2 (scalarProduct nv (mapcar '- c2 p1))
                     d3 (scalarProduct nv (mapcar '- c3 p1))
                     d< (max d0 d1 d2 d3)
                     d> (min d0 d1 d2 d3)
                     ; distances from slicing plane
                     ie (assoc 70 id)
                     ; visibility of edges
                  )
                  (if
                     (and (<= tol d<) (>= (- tol) d>))
                     ; slice only if slicing plane is crossed
                     (xsliceProcessFace)
                     (doNotSlice)
                  )
               )
               (
                  (= "POLYLINE" it)   ; polygon mesh or polyface mesh
                  (setq
                     jn (entnext in)
                     d<
                        (scalarProduct
                           nv
                           (mapcar '- (cdr (assoc 10 (entget jn))) p1)
                        )
                     d> d<
                  )
                  (while
                     (and
                        (setq jt 
                           (assoc
                              70
                              (setq jd (entget (setq jn (entnext jn))))
                           )
                        )
                        (= 64 (logand 64 (cdr jt)))
                     )   ; test all vertices
                     (setq
                        d0
                           (scalarProduct
                              nv
                              (mapcar '- (cdr (assoc 10 jd)) p1)
                           )
                        d< (max d< d0)
                        d> (min d> d0)
                     )
                  )
                  (if
                     (and (<= tol d<) (>= (- tol) d>))
                     ; slice only if slicing plane is crossed
                     (xsliceProcessMesh)
                     (doNotSlice)
                  )
               )
            )
            (setq i# (1+ i#))
         )
         (if (< 0 n#)
            (progn
               (princ
                  (strcat
                     (if ger
                        "Die Kappebene schneidet "
                        "Slicing plane does not intersect "
                     )
                     (cond
                        (
                           (=  1 s#)
                           (if ger
                              "das gewhlte Objekt nicht."
                              "the selected object."
                           )
                        )
                        (
                           (= n# s#)
                           (if ger
                              "die gewhlten Objekte nicht."
                              "the selected objects."
                           )
                        )
                        (
                           t
                           (strcat
                              (itoa n#)
                              (if ger
                                 " der gewhlten Objekte nicht."
                                 " of the selected objects."
                              )
                           )
                        )
                     )
                  )
               )
               (if (< 0 u#)
                  (progn
                     (initget (if ger "Ja Nein _Yes No" "Yes No"))
                     (if
                        (/= "No"
                           (getkword
                              (strcat "\n"
                                 (if (= 1 u#)
                                    (if (=  1 n#)
                                       (if ger
                                          "Es liegt"
                                          "It lies"
                                       )
                                       (if ger
                                          "1 davon liegt"
                                          "1 of them lies"
                                       )
                                    )
                                    (if (= u# n#)
                                       (if ger
                                          "Sie liegen"
                                          "They lie"
                                       )
                                       (strcat
                                          (itoa u#)
                                          (if ger
                                             " davon liegen"
                                             " of them lie"
                                          )
                                       )
                                    )
                                 )
                                 (if ger
                                    " auf der unerwnschten Seite."
                                    " on the undesired side."
                                 )
                                 (if ger
                                    (if r14
                                       " Lschen? <Ja>/Nein: "
                                       " Lschen? [Ja/Nein] <Ja>: "
                                    )
                                    (if r14
                                       " Delete? <Yes>/No: "
                                       " Delete? [Yes/No] <Yes>: "
                                    )
                                 )
                              )
                           )
                        )
                        (command "_.erase" u "")
                        (command "_.regen")   ; unhighlight objects
                     )
                  )
               )
            )
         )
      )
   )
)



;;;   Unterprogramme 2. Ordnung fr xsliceProcess
;;;   2nd order subroutines for xsliceProcess


(defun xsliceProcessLine
   ( )

   (setq ip (interPoint (cdr i0) (cdr i1) d0 d1))
   (if (minusp d0)
      (if  d-
         (progn   ; start point is on desired side
            (entmod (subst (cons 11 ip) i1 id))
            (if b* (progn (ssadd in b-) (setq v- (+ 2 v-))))
            (if d+
               (progn   ; other side also desired
                  (entmake (subst (cons 10 ip) i0 id))
                  (if b* (progn (ssadd (entlast) b+) (setq v+ (+ 2 v+))))
               )
            )
         )
         (progn   ; only end point is on desired side
            (entmod (subst (cons 10 ip) i0 id))
            (if b* (progn (ssadd in b+) (setq v+ (+ 2 v+))))
         )
      )
      (if  d+
         (progn   ; start point is on desired side
            (entmod (subst (cons 11 ip) i1 id))
            (if b* (progn (ssadd in b+) (setq v+ (+ 2 v+))))
            (if d-
               (progn   ; other side also desired
                  (entmake (subst (cons 10 ip) i0 id))
                  (if b* (progn (ssadd (entlast) b-) (setq v- (+ 2 v-))))
               )
            )
         )
         (progn   ; only end point is on desired side
            (entmod (subst (cons 10 ip) i0 id))
            (if b* (progn (ssadd in b-) (setq v- (+ 2 v-))))
         )
      )
   )   ; b* is not nil if the line is a component of a dismantled mesh
)



(defun xsliceProcessRay
   ( )

   (setq ip
      (interPoint
         (cdr i0)
         (mapcar '+ (cdr i0) (cdr i1))
         d0
         (+ d0 d1)
      )
   )
   (if (if (minusp d0) d- d+)
      (progn   ; start point is on desired side
         (entmake
            (subst
               '(0 . "LINE")
               '(0 . "RAY")
               (subst
                  '(100 . "AcDbLine")
                  '(100 . "AcDbRay")
                  (subst (cons 11 ip) i1 id)
               )   ; line between start point and intersection point
            )
         )
         (if (if (minusp d0) d+ d-)
            (entmod (subst (cons 10 ip) i0 id))   ; both sides desired
            (entdel in)   ; only side of start point desired
         )
      )
      (entmod (subst (cons 10 ip) i0 id))   ; start point is on undesired side
   )
)



(defun xsliceProcessXline
   ( )

   (setq id
      (subst
         '(0 . "RAY")
         '(0 . "XLINE")
         (subst
            '(100 . "AcDbRay")
            '(100 . "AcDbXline")
            id
         )
      )
   )
   (if (not (equal 0.0 d0 tol))   ; If the "center" point
      (setq id                    ; is not on the plane yet ...
         (subst
            (cons
               10                 ; ... the intersection point
               (interPoint        ; has to be calculated.
                  (cdr i0)
                  (mapcar '+ (cdr i0) (cdr i1))
                  d0
                  (+ d0 d1)
               )
            )
            i0
            id
         )
      )
   )
   (if (if (minusp d1) d- d+)   ; original direction
      (entmake id)
   )
   (if (if (minusp d1) d+ d-)   ; opposite direction
      (entmake (subst (cons 11 (mapcar '- (cdr i1))) i1 id))
   )
   (entdel in)
)



(defun xsliceProcessFace
   ( )

   ; 3d-Flchen: R14 - zwei Dreiecke, die sich an der "Diagonalen"
   ;                   vom ersten zum dritten Eckpunkt berhren
   ;             R2000 - ... an der anderen Diagonalen ...
   ; Aufschlsseln der Flle nach der Lage der Eckpunkte bezglich der Ebene
   ; [Es ist zwar mglich, die einzelnen Flle z. B. mittels verschachtelter
   ;  if-Anweisungen weiter zusammenzufassen und den Programmcode dadurch zu
   ;  verkrzen; jedoch erhht dies wahrscheinlich die Bearbeitungszeiten.]

   ; 3D faces: R14 - two triangles touching one another along the "diagonal"
   ;                 from the first to the third corner
   ;           R2000 - ... the other diagonal ...
   ; Division of cases by situation of corners relative to slicing plane
   ; [If cases were combined e. g. by nested "if" functions, the program code
   ;  would be shorter but running time would probably be longer.]

   (if r14
      (cond   ; AutoCAD 14


         ;;  Flle 1 bis 3: zweiter oder vierter Eckpunkt abgeschnitten
         ;;  Cases 1 to 3: second or fourth corner cut off

         (   ;  1A   +-++  +-+o      *14*
            (and (<= tol d0) (>= (- tol) d1) (<= tol d2) (< (- tol) d3))
            (setq
               ip (interPoint c0 c1 d0 d1)
               iq (interPoint c1 c2 d1 d2)
            )
            (if d+
               (progn   ; pentangle composed of a quadrangle and a triangle
                  (modFace-1 nil   t)
                  (modFace nil  c0 nil nil 13 2   t   t)
                  (if d-
                     (modFace  ip nil  iq  iq  7 0   t nil)
                  )
               )
               (modFace  ip nil  iq  iq  7 0 nil nil)
            )
         )
         (   ;  1B   -+--  -+-o      *14*
            (and (>= (- tol) d0) (<= tol d1) (>= (- tol) d2) (> tol d3))
            (setq
               ip (interPoint c0 c1 d0 d1)
               iq (interPoint c1 c2 d1 d2)
            )
            (if d-
               (progn   ; pentangle composed of a quadrangle and a triangle
                  (modFace-1 nil nil)
                  (modFace nil  c0 nil nil 13 2   t nil)
                  (if d+
                     (modFace  ip nil  iq  iq  7 0   t   t)
                  )
               )
               (modFace  ip nil  iq  iq  7 0 nil   t)
            )
         )
         (   ;  2A   +++-  +o+-      *14*
            (and (<= tol d0) (< (- tol) d1) (<= tol d2) (>= (- tol) d3))
            (setq
               ir (interPoint c2 c3 d2 d3)
               is (interPoint c3 c0 d3 d0)
            )
            (if d+
               (progn   ; pentangle composed of a quadrangle and a triangle
                  (modFace-3 nil   t)
                  (modFace nil nil nil  c2  7 8   t   t)
                  (if d-
                     (modFace  is  is  ir nil 13 0   t nil)
                  )
               )
               (modFace  is  is  ir nil 13 0 nil nil)
            )
         )
         (   ;  2B   ---+  -o-+      *14*
            (and (>= (- tol) d0) (> tol d1) (>= (- tol) d2) (<= tol d3))
            (setq
               ir (interPoint c2 c3 d2 d3)
               is (interPoint c3 c0 d3 d0)
            )
            (if d-
               (progn   ; pentangle composed of a quadrangle and a triangle
                  (modFace-3 nil nil)
                  (modFace nil nil nil  c2  7 8   t nil)
                  (if d+
                     (modFace  is  is  ir nil 13 0   t   t)
                  )
               )
               (modFace  is  is  ir nil 13 0 nil   t)
            )
         )
         (   ;  3A   +-+-      *14*
            (and (<= tol d0) (>= (- tol) d1) (<= tol d2) (>= (- tol) d3))
            (setq
               ip (interPoint c0 c1 d0 d1)
               iq (interPoint c1 c2 d1 d2)
               ir (interPoint c2 c3 d2 d3)
               is (interPoint c3 c0 d3 d0)
            )
            (if d+
               (progn   ; hexangle composed of two quadrangles
                  (modFace-1 nil   t)
                  (modFace-3   t   t)
                  (if d-
                     (progn
                        (modFace  ip nil  iq  iq  7 0   t nil)
                        (modFace  is  is  ir nil 13 0   t nil)
                     )
                  )
               )
               (progn
                  (modFace  ip nil  iq  iq  7 0 nil nil)
                  (modFace  is  is  ir nil 13 0   t nil)
               )
            )
         )
         (   ;  3B   -+-+      *14*
            (and (>= (- tol) d0) (<= tol d1) (>= (- tol) d2) (<= tol d3))
            (setq
               ip (interPoint c0 c1 d0 d1)
               iq (interPoint c1 c2 d1 d2)
               ir (interPoint c2 c3 d2 d3)
               is (interPoint c3 c0 d3 d0)
            )
            (if d-
               (progn   ; hexangle composed of two quadrangles
                  (modFace-1 nil nil)
                  (modFace-3   t nil)
                  (if d+
                     (progn
                        (modFace  ip nil  iq  iq  7 0   t   t)
                        (modFace  is  is  ir nil 13 0   t   t)
                     )
                  )
               )
               (progn
                  (modFace  ip nil  iq  iq  7 0 nil   t)
                  (modFace  is  is  ir nil 13 0   t   t)
               )
            )
         )


         ;;  Flle 4 bis 7: kein Eckpunkt auf der Kappebene
         ;;  Cases 4 to 7: no corner on the slicing plane

         (   ;  4A   +--+      *14*
            (and (<= tol d0) (>= (- tol) d1) (>= (- tol) d2) (<= tol d3))
            (setq
               ip (interPoint c0 c1 d0 d1)
               iq (interPoint c2 c3 d2 d3)
            )
            (if (coplanar c0 c1 c2 c3)
               (if d+
                  (progn
                     (modFace nil  ip  iq nil 13 0 nil   t)
                     (if d-
                        (modFace  ip nil nil  iq  7 0   t nil)
                     )
                  )
                  (modFace  ip nil nil  iq  7 0 nil nil)
               )
               (progn   ; [points are not coplanar, faces have to be split]
                  (setq ir (interPoint c0 c2 d0 d2))
                  (if d+
                     (progn
                        (modFace nil  ip  ir  ir  5 8 nil   t)
                        (modFace nil  ir  iq nil 12 1   t   t)
                        (if d-
                           (progn
                              (modFace  ip nil nil  ir  3 4   t nil)
                              (modFace  ir  ir nil  iq  5 2   t nil)
                           )
                        )
                     )
                     (progn
                        (modFace  ip nil nil  ir  3 4 nil nil)
                        (modFace  ir  ir nil  iq  5 2   t nil)
                     )
                  )
               )
            )
         )
         (   ;  4B   -++-      *14*
            (and (>= (- tol) d0) (<= tol d1) (<= tol d2) (>= (- tol) d3))
            (setq
               ip (interPoint c0 c1 d0 d1)
               iq (interPoint c2 c3 d2 d3)
            )
            (if (coplanar c0 c1 c2 c3)
               (if d-
                  (progn
                     (modFace nil  ip  iq nil 13 0 nil nil)
                     (if d+
                        (modFace  ip nil nil  iq  7 0   t   t)
                     )
                  )
                  (modFace  ip nil nil  iq  7 0 nil   t)
               )
               (progn   ; [points are not coplanar, faces have to be split]
                  (setq ir (interPoint c0 c2 d0 d2))
                  (if d-
                     (progn
                        (modFace nil  ip  ir  ir  5 8 nil nil)
                        (modFace nil  ir  iq nil 12 1   t nil)
                        (if d+
                           (progn
                              (modFace  ip nil nil  ir  3 4   t   t)
                              (modFace  ir  ir nil  iq  5 2   t   t)
                           )
                        )
                     )
                     (progn
                        (modFace  ip nil nil  ir  3 4 nil   t)
                        (modFace  ir  ir nil  iq  5 2   t   t)
                     )
                  )
               )
            )
         )
         (   ;  5A   ++--      *14*
            (and (<= tol d0) (<= tol d1) (>= (- tol) d2) (>= (- tol) d3))
            (setq
               ip (interPoint c1 c2 d1 d2)
               iq (interPoint c3 c0 d3 d0)
            )
            (if (coplanar c0 c1 c2 c3)
               (if d+
                  (progn
                     (modFace nil nil  ip  iq 11 0 nil   t)
                     (if d-
                        (modFace  iq  ip nil nil 14 0   t nil)
                     )
                  )
                  (modFace  iq  ip nil nil 14 0 nil nil)
               )
               (progn   ; [points are not coplanar, faces have to be split]
                  (setq ir (interPoint c0 c2 d0 d2))
                  (if d+
                     (progn
                        (modFace nil  ir  ir  iq 10 1 nil   t)
                        (modFace nil nil  ip  ir  3 8   t   t)
                        (if d-
                           (progn
                              (modFace  ir  ip nil  ir 10 4   t nil)
                              (modFace  iq  ir nil nil 12 2   t nil)
                           )
                        )
                     )
                     (progn
                        (modFace  ir  ip nil  ir 10 4 nil nil)
                        (modFace  iq  ir nil nil 12 2   t nil)
                     )
                  )
               )
            )
         )
         (   ;  5B   --++      *14*
            (and (>= (- tol) d0) (>= (- tol) d1) (<= tol d2) (<= tol d3))
            (setq
               ip (interPoint c1 c2 d1 d2)
               iq (interPoint c3 c0 d3 d0)
            )
            (if (coplanar c0 c1 c2 c3)
               (if d-
                  (progn
                     (modFace nil nil  ip  iq 11 0 nil nil)
                     (if d+
                        (modFace  iq  ip nil nil 14 0   t   t)
                     )
                  )
                  (modFace  iq  ip nil nil 14 0 nil   t)
               )
               (progn   ; [points are not coplanar, faces have to be split]
                  (setq ir (interPoint c0 c2 d0 d2))
                  (if d-
                     (progn
                        (modFace nil  ir  ir  iq 10 1 nil nil)
                        (modFace nil nil  ip  ir  3 8   t nil)
                        (if d+
                           (progn
                              (modFace  ir  ip nil  ir 10 4   t   t)
                              (modFace  iq  ir nil nil 12 2   t   t)
                           )
                        )
                     )
                     (progn
                        (modFace  ir  ip nil  ir 10 4 nil   t)
                        (modFace  iq  ir nil nil 12 2   t   t)
                     )
                  )
               )
            )
         )
         (   ;  6A   -+++      *14*
            (and (>= (- tol) d0) (<= tol d1) (<= tol d2) (<= tol d3))
            (setq
               ip (interPoint c0 c1 d0 d1)
               iq (interPoint c3 c0 d3 d0)
               ir (interPoint c0 c2 d0 d2)
            )
            (if d+
               (progn
                  (modFace  ip nil nil  ir  3 4 nil   t)
                  (modFace  iq  ir nil nil 12 2   t   t)
                  (if d-
                     (modFace nil  ip  ir  iq  9 0   t nil)
                  )
               )
               (modFace nil  ip  ir  iq  9 0 nil nil)
            )
         )
         (   ;  6B   +---      *14*
            (and (<= tol d0) (>= (- tol) d1) (>= (- tol) d2) (>= (- tol) d3))
            (setq
               ip (interPoint c0 c1 d0 d1)
               iq (interPoint c3 c0 d3 d0)
               ir (interPoint c0 c2 d0 d2)
            )
            (if d-
               (progn
                  (modFace  ip nil nil  ir  3 4 nil nil)
                  (modFace  iq  ir nil nil 12 2   t nil)
                  (if d+
                     (modFace nil  ip  ir  iq  9 0   t   t)
                  )
               )
               (modFace nil  ip  ir  iq  9 0 nil   t)
            )
         )
         (   ;  7A   ++-+      *14*
            (and (<= tol d0) (<= tol d1) (>= (- tol) d2) (<= tol d3))
            (setq
               ip (interPoint c1 c2 d1 d2)
               iq (interPoint c2 c3 d2 d3)
               ir (interPoint c0 c2 d0 d2)
            )
            (if d+
               (progn
                  (modFace nil  ir  iq nil 12 1 nil   t)
                  (modFace nil nil  ip  ir  3 8   t   t)
                  (if d-
                     (modFace  ir  ip nil  iq  6 0   t nil)
                  )
               )
               (modFace  ir  ip nil  iq  6 0 nil nil)
            )
         )
         (   ;  7B   --+-      *14*
            (and (>= (- tol) d0) (>= (- tol) d1) (<= tol d2) (>= (- tol) d3))
            (setq
               ip (interPoint c1 c2 d1 d2)
               iq (interPoint c2 c3 d2 d3)
               ir (interPoint c0 c2 d0 d2)
            )
            (if d-
               (progn
                  (modFace nil  ir  iq nil 12 1 nil nil)
                  (modFace nil nil  ip  ir  3 8   t nil)
                  (if d+
                     (modFace  ir  ip nil  iq  6 0   t   t)
                  )
               )
               (modFace  ir  ip nil  iq  6 0 nil   t)
            )
         )


         ;;  Flle 8 und 9:
         ;;  zwei nicht aufeinander folgende Eckpunkte auf der Kappebene

         ;;  Cases 8 and 9:
         ;;  two non-successive corners on the slicing plane

         (   ;  8   o+o-  o-o+      *14*
            (and (equal 0.0 d0 tol) (equal 0.0 d2 tol))
            (if (minusp d3)
               (if  d+   ; [8A o+o-]
                  (progn
                     (modFace nil nil nil  c2  7 0 nil   t)
                     (if d-
                        (modFace nil  c0 nil nil 13 0   t nil)
                     )
                  )
                  (modFace nil  c0 nil nil 13 0 nil nil)
               )
               (if  d-   ; [8B o-o+]
                  (progn
                     (modFace nil nil nil  c2  7 0 nil nil)
                     (if d+
                        (modFace nil  c0 nil nil 13 0   t   t)
                     )
                  )
                  (modFace nil  c0 nil nil 13 0 nil   t)
               )
            )
         )
         (   ;  9   +o-o  -o+o      *14*
            (and (equal 0.0 d1 tol) (equal 0.0 d3 tol))
            (setq ip (interPoint c0 c2 d0 d2))
            (if (minusp d2)
               (if d+   ; [9A +o-o]
                  (progn
                     (modFace nil nil  ip nil  9 0 nil   t)
                     (if d-
                        (modFace  ip nil nil nil  6 0   t nil)
                     )
                  )
                  (modFace  ip nil nil nil  6 0 nil nil)
               )
               (if d-   ; [9B -o+o]
                  (progn
                     (modFace nil nil  ip nil  9 0 nil nil)
                     (if d+
                        (modFace  ip nil nil nil  6 0   t   t)
                     )
                  )
                  (modFace  ip nil nil nil  6 0 nil   t)
               )
            )
         )


         ;;  Flle 10 bis 12: erster Eckpunkt auf der Kappebene
         ;;  Cases 10 to 12: first corner on the slicing plane

         (
            (equal 0.0 d0 tol)
            (cond
               (   ; 10A   o++-  oo+-      *14*
                  (and (< (- tol) d1) (<= tol d2) (>= (- tol) d3))
                  (setq ip (interPoint c2 c3 d2 d3))
                  (if d+
                     (progn
                        (modFace nil nil nil  ip  7 0 nil   t)
                        (if d-
                           (modFace nil  ip  ip nil 14 0   t nil)
                        )
                     )
                     (modFace nil  ip  ip nil 14 0 nil nil)
                  )
               )
               (   ; 10B   o--+  oo-+      *14*
                  (and (> tol d1) (>= (- tol) d2) (<= tol d3))
                  (setq ip (interPoint c2 c3 d2 d3))
                  (if d-
                     (progn
                        (modFace nil nil nil  ip  7 0 nil nil)
                        (if d+
                           (modFace nil  ip  ip nil 14 0   t   t)
                        )
                     )
                     (modFace nil  ip  ip nil 14 0 nil   t)
                  )
               )
               (   ; 11A   o+-+      *14*
                  (and (<= tol d1) (>= (- tol) d2) (<= tol d3))
                  (setq
                     ip (interPoint c1 c2 d1 d2)
                     iq (interPoint c2 c3 d2 d3)
                  )
                  (if d+
                     (progn
                        (modFace nil nil  ip  ip  7 0 nil   t)
                        (modFace nil  iq  iq nil 14 0   t   t)
                        (if d-
                           (modFace nil  ip nil  iq  6 0   t nil)
                        )
                     )
                     (modFace nil  ip nil  iq  6 0 nil nil)
                  )
               )
               (   ; 11B   o-+-      *14*
                  (and (>= (- tol) d1) (<= tol d2) (>= (- tol) d3))
                  (setq
                     ip (interPoint c1 c2 d1 d2)
                     iq (interPoint c2 c3 d2 d3)
                  )
                  (if d-
                     (progn
                        (modFace nil nil  ip  ip  7 0 nil nil)
                        (modFace nil  iq  iq nil 14 0   t nil)
                        (if d+
                           (modFace nil  ip nil  iq  6 0   t   t)
                        )
                     )
                     (modFace nil  ip nil  iq  6 0 nil   t)
                  )
               )
               (   ; 12A   o-++  o-+o      *14*
                  (and (>= (- tol) d1) (<= tol d2) (< (- tol) d3))
                  (setq ip (interPoint c1 c2 d1 d2))
                  (if d+
                     (progn
                        (modFace nil  ip nil nil 14 0 nil   t)
                        (if d-
                           (modFace nil nil  ip  ip  7 0   t nil)
                        )
                     )
                     (modFace nil nil  ip  ip  7 0 nil nil)
                  )
               )
               (   ; 12B   o+--  o+-o      *14*
                  (and (<= tol d1) (>= (- tol) d2) (> tol d3))
                  (setq ip (interPoint c1 c2 d1 d2))
                  (if d-
                     (progn
                        (modFace nil  ip nil nil 14 0 nil nil)
                        (if d+
                           (modFace nil nil  ip  ip  7 0   t   t)
                        )
                     )
                     (modFace nil nil  ip  ip  7 0 nil   t)
                  )
               )
            )
         )


         ;;  Flle 13 bis 15: dritter Eckpunkt auf der Kappebene
         ;;  Cases 13 to 15: third corner on the slicing plane

         (
            (equal 0.0 d2 tol)
            (cond
               (   ; 13A   +-o+  +-oo      *14*
                  (and (< (- tol) d3) (<= tol d0) (>= (- tol) d1))
                  (setq ip (interPoint c0 c1 d0 d1))
                  (if d+
                     (progn
                        (modFace nil  ip nil nil 13 0 nil   t)
                        (if d-
                           (modFace  ip nil nil  ip 11 0   t nil)
                        )
                     )
                     (modFace  ip nil nil  ip 11 0 nil nil)
                  )
               )
               (   ; 13B   -+o-  -+oo      *14*
                  (and (> tol d3) (>= (- tol) d0) (<= tol d1))
                  (setq ip (interPoint c0 c1 d0 d1))
                  (if d-
                     (progn
                        (modFace nil  ip nil nil 13 0 nil nil)
                        (if d+
                           (modFace  ip nil nil  ip 11 0   t   t)
                        )
                     )
                     (modFace  ip nil nil  ip 11 0 nil   t)
                  )
               )
               (   ; 14A   -+o+      *14*
                  (and (<= tol d3) (>= (- tol) d0) (<= tol d1))
                  (setq
                     ip (interPoint c3 c0 d3 d0)
                     iq (interPoint c0 c1 d0 d1)
                  )
                  (if d+
                     (progn
                        (modFace  ip  ip nil nil 13 0 nil   t)
                        (modFace  iq nil nil  iq 11 0   t   t)
                        (if d-
                           (modFace nil  ip nil  iq  9 0   t nil)
                        )
                     )
                     (modFace nil  ip nil  iq  9 0 nil nil)
                  )
               )
               (   ; 14B   +-o-      *14*
                  (and (>= (- tol) d3) (<= tol d0) (>= (- tol) d1))
                  (setq
                     ip (interPoint c3 c0 d3 d0)
                     iq (interPoint c0 c1 d0 d1)
                  )
                  (if d-
                     (progn
                        (modFace  ip  ip nil nil 13 0 nil nil)
                        (modFace  iq nil nil  iq 11 0   t nil)
                        (if d+
                           (modFace nil  ip nil  iq  9 0   t   t)
                        )
                     )
                     (modFace nil  ip nil  iq  9 0 nil   t)
                  )
               )
               (   ; 15A   ++o-  +oo-      *14*
                  (and (>= (- tol) d3) (<= tol d0) (< (- tol) d1))
                  (setq ip (interPoint c3 c0 d3 d0))
                  (if d+
                     (progn
                        (modFace nil nil nil  ip 11 0 nil   t)
                        (if d-
                           (modFace  ip  ip nil nil 13 0   t nil)
                        )
                     )
                     (modFace  ip  ip nil nil 13 0 nil nil)
                  )
               )
               (   ; 15B   --o+  -oo+      *14*
                  (and (<= tol d3) (>= (- tol) d0) (> tol d1))
                  (setq ip (interPoint c3 c0 d3 d0))
                  (if d-
                     (progn
                        (modFace nil nil nil  ip 11 0 nil nil)
                        (if d+
                           (modFace  ip  ip nil nil 13 0   t   t)
                        )
                     )
                     (modFace  ip  ip nil nil 13 0 nil   t)
                  )
               )
            )
         )


         ;;  Flle 16 und 17: zweiter Eckpunkt auf der Kappebene
         ;;  Cases 16 and 17: second corner on the slicing plane

         (
            (equal 0.0 d1 tol)
            (cond
               (   ; 16A   -o++      *14*
                  (and (<= tol d2) (<= tol d3) (>= (- tol) d0))
                  (setq
                     ip (interPoint c3 c0 d3 d0)
                     iq (interPoint c0 c2 d0 d2)
                  )
                  (if d+
                     (progn
                        (if (coplanar c0 c1 c2 c3)
                           (modFace  ip nil nil nil 14 0 nil   t)
                           (progn
                              (modFace  iq nil nil  iq 10 4 nil   t)
                              (modFace  ip  iq nil nil 12 2   t   t)
                           )
                        )
                        (if d-
                           (modFace nil nil  iq  ip  9 0   t nil)
                        )
                     )
                     (modFace nil nil  iq  ip  9 0 nil nil)
                  )
               )
               (   ; 16B   +o--      *14*
                  (and (>= (- tol) d2) (>= (- tol) d3) (<= tol d0))
                  (setq
                     ip (interPoint c3 c0 d3 d0)
                     iq (interPoint c0 c2 d0 d2)
                  )
                  (if d-
                     (progn
                        (if (coplanar c0 c1 c2 c3)
                           (modFace  ip nil nil nil 14 0 nil nil)
                           (progn
                              (modFace  iq nil nil  iq 10 4 nil nil)
                              (modFace  ip  iq nil nil 12 2   t nil)
                           )
                        )
                        (if d+
                           (modFace nil nil  iq  ip  9 0    t   t)
                        )
                     )
                     (modFace nil nil  iq  ip  9 0 nil   t)
                  )
               )
               (   ; 17A   +o-+      *14*
                  (and (>= (- tol) d2) (<= tol d3) (<= tol d0))
                  (setq
                     ip (interPoint c2 c3 d2 d3)
                     iq (interPoint c0 c2 d0 d2)
                  )
                  (if d+
                     (progn
                        (if (coplanar c0 c1 c2 c3)
                           (modFace nil nil  ip nil 13 0 nil   t)
                           (progn
                              (modFace nil nil  iq  iq  5 8 nil   t)
                              (modFace nil  iq  ip nil 12 1   t   t)
                           )
                        )
                        (if d-
                           (modFace  iq nil nil  ip  6 0   t nil)
                        )
                     )
                     (modFace  iq nil nil  ip  6 0 nil nil)
                  )
               )
               (   ; 17B   -o+-      *14*
                  (and (<= tol d2) (>= (- tol) d3) (>= (- tol) d0))
                  (setq
                     ip (interPoint c2 c3 d2 d3)
                     iq (interPoint c0 c2 d0 d2)
                  )
                  (if d-
                     (progn
                        (if (coplanar c0 c1 c2 c3)
                           (modFace nil nil  ip nil 13 0 nil nil)
                           (progn
                              (modFace nil nil  iq  iq  5 8 nil nil)
                              (modFace nil  iq  ip nil 12 1   t nil)
                           )
                        )
                        (if d+
                           (modFace  iq nil nil  ip  6 0   t   t)
                        )
                     )
                     (modFace  iq nil nil  ip  6 0 nil   t)
                  )
               )
            )
         )


         ;;  Flle 18 und 19: vierter Eckpunkt auf der Kappebene
         ;;  Cases 18 and 19: fourth corner on the slicing plane

         (
            (equal 0.0 d3 tol)
            (cond
               (   ; 18A   ++-o      *14*
                  (and (<= tol d0) (<= tol d1) (>= (- tol) d2))
                  (setq
                     ip (interPoint c1 c2 d1 d2)
                     iq (interPoint c0 c2 d0 d2)
                  )
                  (if d+
                     (progn
                        (if (coplanar c0 c1 c2 c3)
                           (modFace nil nil  ip nil 11 0 nil   t)
                           (progn
                              (modFace nil  iq  iq nil 10 1 nil   t)
                              (modFace nil nil  ip  iq  3 8   t   t)
                           )
                        )
                        (if d-
                           (modFace  iq  ip nil nil  6 0   t nil)
                        )
                     )
                     (modFace  iq  ip nil nil  6 0 nil nil)
                  )
               )
               (   ; 18B   --+o      *14*
                  (and (>= (- tol) d0) (>= (- tol) d1) (<= tol d2))
                  (setq
                     ip (interPoint c1 c2 d1 d2)
                     iq (interPoint c0 c2 d0 d2)
                  )
                  (if d-
                     (progn
                        (if (coplanar c0 c1 c2 c3)
                           (modFace nil nil  ip nil 11 0 nil nil)
                           (progn
                              (modFace nil  iq  iq nil 10 1 nil nil)
                              (modFace nil nil  ip  iq  3 8   t nil)
                           )
                        )
                        (if d+
                           (modFace  iq  ip nil nil  6 0   t   t)
                        )
                     )
                     (modFace  iq  ip nil nil  6 0 nil   t)
                  )
               )
               (   ; 19A   -++o      *14*
                  (and (>= (- tol) d0) (<= tol d1) (<= tol d2))
                  (setq
                     ip (interPoint c0 c1 d0 d1)
                     iq (interPoint c0 c2 d0 d2)
                  )
                  (if d+
                     (progn
                        (if (coplanar c0 c1 c2 c3)
                           (modFace  ip nil nil nil  7 0 nil   t)
                           (progn
                              (modFace  iq  iq nil nil  5 2 nil   t)
                              (modFace  ip nil nil  iq  3 4   t   t)
                           )
                        )
                        (if d-
                           (modFace nil  ip  iq nil  9 0   t nil)
                        )
                     )
                     (modFace nil  ip  iq nil  9 0 nil nil)
                  )
               )
               (   ; 19B   +--o      *14*
                  (and (<= tol d0) (>= (- tol) d1) (>= (- tol) d2))
                  (setq
                     ip (interPoint c0 c1 d0 d1)
                     iq (interPoint c0 c2 d0 d2)
                  )
                  (if d-
                     (progn
                        (if (coplanar c0 c1 c2 c3)
                           (modFace  ip nil nil nil  7 0 nil nil)
                           (progn
                              (modFace  iq  iq nil nil  5 2 nil nil)
                              (modFace  ip nil nil  iq  3 4   t nil)
                           )
                        )
                        (if d+
                           (modFace nil  ip  iq nil  9 0   t   t)
                        )
                     )
                     (modFace nil  ip  iq nil  9 0 nil   t)
                  )
               )
            )
         )
      )
      (cond   ; AutoCAD 2000


         ;;  Flle 1 bis 3: erster oder dritter Eckpunkt abgeschnitten
         ;;  Cases 1 to 3: first or third corner cut off

         (   ;  1A   -+++  -+o+
            (and (>= (- tol) d0) (<= tol d1) (< (- tol) d2) (<= tol d3))
            (setq
               ip (interPoint c3 c0 d3 d0)
               iq (interPoint c0 c1 d0 d1)
            )
            (if d+
               (progn   ; pentangle composed of a quadrangle and a triangle
                  (modFace-0 nil   t)
                  (modFace  c3 nil nil nil 14 1   t   t)
                  (if d-
                     (modFace nil  iq  iq  ip 11 0   t nil)
                  )
               )
               (modFace nil  iq  iq  ip 11 0 nil nil)
            )
         )
         (   ;  1B   +---  +-o-
            (and (<= tol d0) (>= (- tol) d1) (> tol d2) (>= (- tol) d3))
            (setq
               ip (interPoint c3 c0 d3 d0)
               iq (interPoint c0 c1 d0 d1)
            )
            (if d-
               (progn   ; pentangle composed of a quadrangle and a triangle
                  (modFace-0 nil nil)
                  (modFace  c3 nil nil nil  14 1   t nil)
                  (if d+
                     (modFace nil  iq  iq  ip 11 0   t   t)
                  )
               )
               (modFace nil  iq  iq  ip 11 0 nil   t)
            )
         )
         (   ;  2A   ++-+  o+-+
            (and (< (- tol) d0) (<= tol d1) (>= (- tol) d2) (<= tol d3))
            (setq
               ir (interPoint c1 c2 d1 d2)
               is (interPoint c2 c3 d2 d3)
            )
            (if d+
               (progn   ; pentangle composed of a quadrangle and a triangle
                  (modFace-2 nil   t)
                  (modFace nil nil  c1 nil 11 4   t   t)
                  (if d-
                     (modFace  is  ir nil  is 14 0   t nil)
                  )
               )
               (modFace  is  ir nil  is 14 0 nil nil)
            )
         )
         (   ;  2B   --+-  o-+-
            (and (> tol d0) (>= (- tol) d1) (<= tol d2) (>= (- tol) d3))
            (setq
               ir (interPoint c1 c2 d1 d2)
               is (interPoint c2 c3 d2 d3)
            )
            (if d-
               (progn   ; pentangle composed of a quadrangle and a triangle
                  (modFace-2 nil nil)
                  (modFace nil nil  c1 nil 11 4   t nil)
                  (if d+
                     (modFace  is  ir nil  is 14 0   t   t)
                  )
               )
               (modFace  is  ir nil  is 14 0 nil   t)
            )
         )
         (   ;  3A   -+-+
            (and (>= (- tol) d0) (<= tol d1) (>= (- tol) d2) (<= tol d3))
            (setq
               ip (interPoint c3 c0 d3 d0)
               iq (interPoint c0 c1 d0 d1)
               ir (interPoint c1 c2 d1 d2)
               is (interPoint c2 c3 d2 d3)
            )
            (if d+
               (progn   ; hexangle composed of two quadrangles
                  (modFace-0 nil   t)
                  (modFace-2   t   t)
                  (if d-
                     (progn
                        (modFace nil  iq  iq  ip 11 0   t nil)
                        (modFace  is  ir nil  is 14 0   t nil)
                     )
                  )
               )
               (progn
                  (modFace nil  iq  iq  ip 11 0 nil nil)
                  (modFace  is  ir nil  is 14 0   t nil)
               )
            )
         )
         (   ;  3B   +-+-
            (and (<= tol d0) (>= (- tol) d1) (<= tol d2) (>= (- tol) d3))
            (setq
               ip (interPoint c3 c0 d3 d0)
               iq (interPoint c0 c1 d0 d1)
               ir (interPoint c1 c2 d1 d2)
               is (interPoint c2 c3 d2 d3)
            )
            (if d-
               (progn   ; hexangle composed of two quadrangles
                  (modFace-0 nil nil)
                  (modFace-2   t nil)
                  (if d+
                     (progn
                        (modFace nil  iq  iq  ip 11 0   t   t)
                        (modFace  is  ir nil  is 14 0   t   t)
                     )
                  )
               )
               (progn
                  (modFace nil  iq  iq  ip 11 0 nil   t)
                  (modFace  is  ir nil  is 14 0   t   t)
               )
            )
         )


         ;;  Flle 4 bis 7: kein Eckpunkt auf der Kappebene
         ;;  Cases 4 to 7: no corner on the slicing plane

         (   ;  4A   --++
            (and (>= (- tol) d0) (>= (- tol) d1) (<= tol d2) (<= tol d3))
            (setq
               ip (interPoint c3 c0 d3 d0)
               iq (interPoint c1 c2 d1 d2)
            )
            (if (coplanar c0 c1 c2 c3)
               (if d+
                  (progn
                     (modFace  ip  iq nil nil 14 0 nil   t)
                     (if d-
                        (modFace nil nil  iq  ip 11 0   t nil)
                     )
                  )
                  (modFace nil nil  iq  ip 11 0 nil nil)
               )
               (progn   ; [points are not coplanar, faces have to be split]
                  (setq ir (interPoint c1 c3 d1 d3))
                  (if d+
                     (progn
                        (modFace  ip  ir  ir nil 10 4 nil   t)
                        (modFace  ir  iq nil nil  6 8   t   t)
                        (if d-
                           (progn
                              (modFace nil nil  ir  ip  9 2   t nil)
                              (modFace  ir nil  iq  ir 10 1   t nil)
                           )
                        )
                     )
                     (progn
                        (modFace nil nil  ir  ip  9 2 nil nil)
                        (modFace  ir nil  iq  ir 10 1   t nil)
                     )
                  )
               )
            )
         )
         (   ;  4B   ++--
            (and (<= tol d0) (<= tol d1) (>= (- tol) d2) (>= (- tol) d3))
            (setq
               ip (interPoint c3 c0 d3 d0)
               iq (interPoint c1 c2 d1 d2)
            )
            (if (coplanar c0 c1 c2 c3)
               (if d-
                  (progn
                     (modFace  ip  iq nil nil 14 0 nil nil)
                     (if d+
                        (modFace nil nil  iq  ip 11 0   t   t)
                     )
                  )
                  (modFace nil nil  iq  ip 11 0 nil   t)
               )
               (progn   ; [points are not coplanar, faces have to be split]
                  (setq ir (interPoint c1 c3 d1 d3))
                  (if d-
                     (progn
                        (modFace  ip  ir  ir nil 10 4 nil nil)
                        (modFace  ir  iq nil nil  6 8   t nil)
                        (if d+
                           (progn
                              (modFace nil nil  ir  ip  9 2   t   t)
                              (modFace  ir nil  iq  ir 10 1   t   t)
                           )
                        )
                     )
                     (progn
                        (modFace nil nil  ir  ip  9 2 nil   t)
                        (modFace  ir nil  iq  ir 10 1   t   t)
                     )
                  )
               )
            )
         )
         (   ;  5A   +--+
            (and (<= tol d3) (<= tol d0) (>= (- tol) d1) (>= (- tol) d2))
            (setq
               ip (interPoint c0 c1 d0 d1)
               iq (interPoint c2 c3 d2 d3)
            )
            (if (coplanar c0 c1 c2 c3)
               (if d+
                  (progn
                     (modFace nil  ip  iq nil 13 0 nil   t)
                     (if d-
                        (modFace  ip nil nil  iq  7 0   t nil)
                     )
                  )
                  (modFace  ip nil nil  iq  7 0 nil nil)
               )
               (progn   ; [points are not coplanar, faces have to be split]
                  (setq ir (interPoint c1 c3 d1 d3))
                  (if d+
                     (progn
                        (modFace  ir  ir  iq nil  5 8 nil   t)
                        (modFace nil  ip  ir nil  9 4   t   t)
                        (if d-
                           (progn
                              (modFace  ip nil  ir  ir  5 2   t nil)
                              (modFace  ir nil nil  iq  6 1   t nil)
                           )
                        )
                     )
                     (progn
                        (modFace  ip nil  ir  ir  5 2 nil nil)
                        (modFace  ir nil nil  iq  6 1   t nil)
                     )
                  )
               )
            )
         )
         (   ;  5B   -++-
            (and (>= (- tol) d0) (<= tol d1) (<= tol d2) (>= (- tol) d3))
            (setq
               ip (interPoint c0 c1 d0 d1)
               iq (interPoint c2 c3 d2 d3)
            )
            (if (coplanar c0 c1 c2 c3)
               (if d-
                  (progn
                     (modFace nil  ip  iq nil 13 0 nil nil)
                     (if d+
                        (modFace  ip nil nil  iq  7 0   t   t)
                     )
                  )
                  (modFace  ip nil nil  iq  7 0 nil   t)
               )
               (progn   ; [points are not coplanar, faces have to be split]
                  (setq ir (interPoint c1 c3 d1 d3))
                  (if d-
                     (progn
                        (modFace  ir  ir  iq nil  5 8 nil nil)
                        (modFace nil  ip  ir nil  9 4   t nil)
                        (if d+
                           (progn
                              (modFace  ip nil  ir  ir  5 2   t   t)
                              (modFace  ir nil nil  iq  6 1   t   t)
                           )
                        )
                     )
                     (progn
                        (modFace  ip nil  ir  ir  5 2 nil   t)
                        (modFace  ir nil nil  iq  6 1   t   t)
                     )
                  )
               )
            )
         )
         (   ;  6A   +++-
            (and (<= tol d0) (<= tol d1) (<= tol d2) (>= (- tol) d3))
            (setq
               ip (interPoint c3 c0 d3 d0)
               iq (interPoint c2 c3 d2 d3)
               ir (interPoint c1 c3 d1 d3)
            )
            (if d+
               (progn
                  (modFace nil nil  ir  ip  9 2 nil   t)
                  (modFace  ir nil nil  iq  6 1   t   t)
                  (if d-
                     (modFace  ip  ir  iq nil 12 0   t nil)
                  )
               )
               (modFace  ip  ir  iq nil 12 0 nil nil)
            )
         )
         (   ;  6B   ---+
            (and (>= (- tol) d0) (>= (- tol) d1) (>= (- tol) d2) (<= tol d3))
            (setq
               ip (interPoint c3 c0 d3 d0)
               iq (interPoint c2 c3 d2 d3)
               ir (interPoint c1 c3 d1 d3)
            )
            (if d-
               (progn
                  (modFace nil nil  ir  ip  9 2 nil nil)
                  (modFace  ir nil nil  iq  6 1   t nil)
                  (if d+
                     (modFace  ip  ir  iq nil 12 0   t   t)
                  )
               )
               (modFace  ip  ir  iq nil 12 0 nil   t)
            )
         )
         (   ;  7A   +-++
            (and (<= tol d0) (>= (- tol) d1) (<= tol d2) (<= tol d3))
            (setq
               ip (interPoint c0 c1 d0 d1)
               iq (interPoint c1 c2 d1 d2)
               ir (interPoint c1 c3 d1 d3)
            )
            (if d+
               (progn
                  (modFace  ir  iq nil nil  6 8 nil   t)
                  (modFace nil  ip  ir nil  9 4   t   t)
                  (if d-
                     (modFace  ip nil  iq  ir  3 0   t nil)
                  )
               )
               (modFace  ip nil  iq  ir  3 0 nil nil)
            )
         )
         (   ;  7B   -+--
            (and (>= (- tol) d0) (<= tol d1) (>= (- tol) d2) (>= (- tol) d3))
            (setq
               ip (interPoint c0 c1 d0 d1)
               iq (interPoint c1 c2 d1 d2)
               ir (interPoint c1 c3 d1 d3)
            )
            (if d-
               (progn
                  (modFace  ir  iq nil nil  6 8 nil nil)
                  (modFace nil  ip  ir nil  9 4   t nil)
                  (if d+
                     (modFace  ip nil  iq  ir  3 0   t   t)
                  )
               )
               (modFace  ip nil  iq  ir  3 0 nil   t)
            )
         )


         ;;  Flle 8 und 9:
         ;;  zwei nicht aufeinander folgende Eckpunkte auf der Kappebene

         ;;  Cases 8 and 9:
         ;;  two non-successive corners on the slicing plane

         (   ;  8   +o-o  -o+o
            (and (equal 0.0 d1 tol) (equal 0.0 d3 tol))
            (if (minusp d2)
               (if  d+   ; [8A +o-o]
                  (progn
                     (modFace nil nil  c1 nil 11 0 nil   t)
                     (if d-
                        (modFace  c3 nil nil nil 14 0   t nil)
                     )
                  )
                  (modFace  c3 nil nil nil 14 0 nil nil)
               )
               (if  d-   ; [8B -o+o]
                  (progn
                     (modFace nil nil  c1 nil 11 0 nil nil)
                     (if d+
                        (modFace  c3 nil nil nil 14 0   t   t)
                     )
                  )
                  (modFace  c3 nil nil nil 14 0 nil   t)
               )
            )
         )
         (   ;  9   o-o+  o+o-
            (and (equal 0.0 d0 tol) (equal 0.0 d2 tol))
            (setq ip (interPoint c1 c3 d1 d3))
            (if (minusp d1)
               (if d+   ; [9A o-o+]
                  (progn
                     (modFace nil  ip nil nil 12 0 nil   t)
                     (if d-
                        (modFace nil nil nil  ip  3 0   t nil)
                     )
                  )
                  (modFace nil nil nil  ip  3 0 nil nil)
               )
               (if d-   ; [9B o+o-]
                  (progn
                     (modFace nil  ip nil nil 12 0 nil nil)
                     (if d+
                        (modFace nil nil nil  ip  3 0   t   t)
                     )
                  )
                  (modFace nil nil nil  ip  3 0 nil   t)
               )
            )
         )


         ;;  Flle 10 bis 12: vierter Eckpunkt auf der Kappebene
         ;;  Cases 10 to 12: fourth corner on the slicing plane

         (
            (equal 0.0 d3 tol)
            (cond
               (   ; 10A   ++-o  o+-o
                  (and (< (- tol) d0) (<= tol d1) (>= (- tol) d2))
                  (setq ip (interPoint c1 c2 d1 d2))
                  (if d+
                     (progn
                        (modFace nil nil  ip nil 11 0 nil   t)
                        (if d-
                           (modFace  ip  ip nil nil  7 0   t nil)
                        )
                     )
                     (modFace  ip  ip nil nil  7 0 nil nil)
                  )
               )
               (   ; 10B   --+o  o-+o
                  (and (> tol d0) (>= (- tol) d1) (<= tol d2))
                  (setq ip (interPoint c1 c2 d1 d2))
                  (if d-
                     (progn
                        (modFace nil nil  ip nil 11 0 nil nil)
                        (if d+
                           (modFace  ip  ip nil nil  7 0   t   t)
                        )
                     )
                     (modFace  ip  ip nil nil  7 0 nil   t)
                  )
               )
               (   ; 11A   +-+o
                  (and (<= tol d0) (>= (- tol) d1) (<= tol d2))
                  (setq
                     ip (interPoint c0 c1 d0 d1)
                     iq (interPoint c1 c2 d1 d2)
                  )
                  (if d+
                     (progn
                        (modFace nil  ip  ip nil 11 0 nil   t)
                        (modFace  iq  iq nil nil  7 0   t   t)
                        (if d-
                           (modFace  ip nil  iq nil  3 0   t nil)
                        )
                     )
                     (modFace  ip nil  iq nil  3 0 nil nil)
                  )
               )
               (   ; 11B   -+-o
                  (and (>= (- tol) d0) (<= tol d1) (>= (- tol) d2))
                  (setq
                     ip (interPoint c0 c1 d0 d1)
                     iq (interPoint c1 c2 d1 d2)
                  )
                  (if d-
                     (progn
                        (modFace nil  ip  ip nil 11 0 nil nil)
                        (modFace  iq  iq nil nil  7 0   t nil)
                        (if d+
                           (modFace  ip nil  iq nil  3 0   t   t)
                        )
                     )
                     (modFace  ip nil  iq nil  3 0 nil   t)
                  )
               )
               (   ; 12A   -++o  -+oo
                  (and (>= (- tol) d0) (<= tol d1) (< (- tol) d2))
                  (setq ip (interPoint c0 c1 d0 d1))
                  (if d+
                     (progn
                        (modFace  ip nil nil nil  7 0 nil   t)
                        (if d-
                           (modFace nil  ip  ip nil 11 0   t nil)
                        )
                     )
                     (modFace nil  ip  ip nil 11 0 nil nil)
                  )
               )
               (   ; 12B   +--o  +-oo
                  (and (<= tol d0) (>= (- tol) d1) (> tol d2))
                  (setq ip (interPoint c0 c1 d0 d1))
                  (if d-
                     (progn
                        (modFace  ip nil nil nil  7 0 nil nil)
                        (if d+
                           (modFace nil  ip  ip nil 11 0   t   t)
                        )
                     )
                     (modFace nil  ip  ip nil 11 0 nil   t)
                  )
               )
            )
         )


         ;;  Flle 13 bis 15: zweiter Eckpunkt auf der Kappebene
         ;;  Cases 13 to 15: second corner on the slicing plane

         (
            (equal 0.0 d1 tol)
            (cond
               (   ; 13A   -o++  -oo+
                  (and (< (- tol) d2) (<= tol d3) (>= (- tol) d0))
                  (setq ip (interPoint c3 c0 d3 d0))
                  (if d+
                     (progn
                        (modFace  ip nil nil nil 14 0 nil   t)
                        (if d-
                           (modFace nil nil  ip  ip 13 0   t nil)
                        )
                     )
                     (modFace nil nil  ip  ip 13 0 nil nil)
                  )
               )
               (   ; 13B   +o--  +oo-
                  (and (> tol d2) (>= (- tol) d3) (<= tol d0))
                  (setq ip (interPoint c3 c0 d3 d0))
                  (if d-
                     (progn
                        (modFace  ip nil nil nil 14 0 nil nil)
                        (if d+
                           (modFace nil nil  ip  ip 13 0   t   t)
                        )
                     )
                     (modFace nil nil  ip  ip 13 0 nil   t)
                  )
               )
               (   ; 14A   +o+-
                  (and (<= tol d2) (>= (- tol) d3) (<= tol d0))
                  (setq
                     ip (interPoint c2 c3 d2 d3)
                     iq (interPoint c3 c0 d3 d0)
                  )
                  (if d+
                     (progn
                        (modFace  ip nil nil  ip 14 0 nil   t)
                        (modFace nil nil  iq  iq 13 0   t   t)
                        (if d-
                           (modFace  ip nil  iq nil 12 0   t nil)
                        )
                     )
                     (modFace  ip nil  iq nil 12 0 nil nil)
                  )
               )
               (   ; 14B   -o-+
                  (and (>= (- tol) d2) (<= tol d3) (>= (- tol) d0))
                  (setq
                     ip (interPoint c2 c3 d2 d3)
                     iq (interPoint c3 c0 d3 d0)
                  )
                  (if d-
                     (progn
                        (modFace  ip nil nil  ip 14 0 nil nil)
                        (modFace nil nil  iq  iq 13 0   t nil)
                        (if d+
                           (modFace  ip nil  iq nil 12 0   t   t)
                        )
                     )
                     (modFace  ip nil  iq nil 12 0 nil   t)
                  )
               )
               (   ; 15A   +o-+  oo-+
                  (and (>= (- tol) d2) (<= tol d3) (< (- tol) d0))
                  (setq ip (interPoint c2 c3 d2 d3))
                  (if d+
                     (progn
                        (modFace nil nil  ip nil 13 0 nil   t)
                        (if d-
                           (modFace  ip nil nil  ip 14 0   t nil)
                        )
                     )
                     (modFace  ip nil nil  ip 14 0 nil nil)
                  )
               )
               (   ; 15B   -o+-  oo+-
                  (and (<= tol d2) (>= (- tol) d3) (> tol d0))
                  (setq ip (interPoint c2 c3 d2 d3))
                  (if d-
                     (progn
                        (modFace nil nil  ip nil 13 0 nil nil)
                        (if d+
                           (modFace  ip nil nil  ip 14 0   t   t)
                        )
                     )
                     (modFace  ip nil nil  ip 14 0 nil   t)
                  )
               )
            )
         )


         ;;  Flle 16 und 17: erster Eckpunkt auf der Kappebene
         ;;  Cases 16 and 17: first corner on the slicing plane

         (
            (equal 0.0 d0 tol)
            (cond
               (   ; 16A   o++-
                  (and (<= tol d1) (<= tol d2) (>= (- tol) d3))
                  (setq
                     ip (interPoint c2 c3 d2 d3)
                     iq (interPoint c1 c3 d1 d3)
                  )
                  (if d+
                     (progn
                        (if (coplanar c0 c1 c2 c3)
                           (modFace nil nil nil  ip  7 0 nil   t)
                           (progn
                              (modFace nil nil  iq  iq  5 2 nil   t)
                              (modFace  iq nil nil  ip  6 1   t   t)
                           )
                        )
                        (if d-
                           (modFace nil  iq  ip nil 12 0   t nil)
                        )
                     )
                     (modFace nil  iq  ip nil 12 0 nil nil)
                  )
               )
               (   ; 16B   o--+
                  (and (>= (- tol) d1) (>= (- tol) d2) (<= tol d3))
                  (setq
                     ip (interPoint c2 c3 d2 d3)
                     iq (interPoint c1 c3 d1 d3)
                  )
                  (if d-
                     (progn
                        (if (coplanar c0 c1 c2 c3)
                           (modFace nil nil nil  ip  7 0 nil nil)
                           (progn
                              (modFace nil nil  iq  iq  5 2 nil nil)
                              (modFace  iq nil nil  ip  6 1   t nil)
                           )
                        )
                        (if d+
                           (modFace nil  iq  ip nil 12 0    t   t)
                        )
                     )
                     (modFace nil  iq  ip nil 12 0 nil   t)
                  )
               )
               (   ; 17A   o-++
                  (and (>= (- tol) d1) (<= tol d2) (<= tol d3))
                  (setq
                     ip (interPoint c1 c2 d1 d2)
                     iq (interPoint c1 c3 d1 d3)
                  )
                  (if d+
                     (progn
                        (if (coplanar c0 c1 c2 c3)
                           (modFace nil  ip nil nil 14 0 nil   t)
                           (progn
                              (modFace nil  iq  iq nil 10 4 nil   t)
                              (modFace  iq  ip nil nil  6 8   t   t)
                           )
                        )
                        (if d-
                           (modFace nil nil  ip  iq  3 0   t nil)
                        )
                     )
                     (modFace nil nil  ip  iq  3 0 nil nil)
                  )
               )
               (   ; 17B   o+--
                  (and (<= tol d1) (>= (- tol) d2) (>= (- tol) d3))
                  (setq
                     ip (interPoint c1 c2 d1 d2)
                     iq (interPoint c1 c3 d1 d3)
                  )
                  (if d-
                     (progn
                        (if (coplanar c0 c1 c2 c3)
                           (modFace nil  ip nil nil 14 0 nil nil)
                           (progn
                              (modFace nil  iq  iq nil 10 4 nil nil)
                              (modFace  iq  ip nil nil  6 8   t nil)
                           )
                        )
                        (if d+
                           (modFace nil nil  ip  iq  3 0   t   t)
                        )
                     )
                     (modFace nil nil  ip  iq  3 0 nil   t)
                  )
               )
            )
         )


         ;;  Flle 18 und 19: dritter Eckpunkt auf der Kappebene
         ;;  Cases 18 and 19: third corner on the slicing plane

         (
            (equal 0.0 d2 tol)
            (cond
               (   ; 18A   +-o+
                  (and (<= tol d3) (<= tol d0) (>= (- tol) d1))
                  (setq
                     ip (interPoint c0 c1 d0 d1)
                     iq (interPoint c1 c3 d1 d3)
                  )
                  (if d+
                     (progn
                        (if (coplanar c0 c1 c2 c3)
                           (modFace nil  ip nil nil 13 0 nil   t)
                           (progn
                              (modFace  iq  iq nil nil  5 8 nil   t)
                              (modFace nil  ip  iq nil  9 4   t   t)
                           )
                        )
                        (if d-
                           (modFace  ip nil nil  iq  3 0   t nil)
                        )
                     )
                     (modFace  ip nil nil  iq  3 0 nil nil)
                  )
               )
               (   ; 18B   -+o-
                  (and (>= (- tol) d3) (>= (- tol) d0) (<= tol d1))
                  (setq
                     ip (interPoint c0 c1 d0 d1)
                     iq (interPoint c1 c3 d1 d3)
                  )
                  (if d-
                     (progn
                        (if (coplanar c0 c1 c2 c3)
                           (modFace nil  ip nil nil 13 0 nil nil)
                           (progn
                              (modFace  iq  iq nil nil  5 8 nil nil)
                              (modFace nil  ip  iq nil  9 4   t nil)
                           )
                        )
                        (if d+
                           (modFace  ip nil nil  iq  3 0   t   t)
                        )
                     )
                     (modFace  ip nil nil  iq  3 0 nil   t)
                  )
               )
               (   ; 19A   ++o-
                  (and (>= (- tol) d3) (<= tol d0) (<= tol d1))
                  (setq
                     ip (interPoint c3 c0 d3 d0)
                     iq (interPoint c1 c3 d1 d3)
                  )
                  (if d+
                     (progn
                        (if (coplanar c0 c1 c2 c3)
                           (modFace nil nil nil  ip 11 0 nil   t)
                           (progn
                              (modFace  iq nil nil  iq 10 1 nil   t)
                              (modFace nil nil  iq  ip  9 2   t   t)
                           )
                        )
                        (if d-
                           (modFace  ip  iq nil nil 12 0   t nil)
                        )
                     )
                     (modFace  ip  iq nil nil 12 0 nil nil)
                  )
               )
               (   ; 19B   --o+
                  (and (<= tol d3) (>= (- tol) d0) (>= (- tol) d1))
                  (setq
                     ip (interPoint c3 c0 d3 d0)
                     iq (interPoint c1 c3 d1 d3)
                  )
                  (if d-
                     (progn
                        (if (coplanar c0 c1 c2 c3)
                           (modFace nil nil nil  ip 11 0 nil nil)
                           (progn
                              (modFace  iq nil nil  iq 10 1 nil nil)
                              (modFace nil nil  iq  ip  9 2   t nil)
                           )
                        )
                        (if d+
                           (modFace  ip  iq nil nil 12 0   t   t)
                        )
                     )
                     (modFace  ip  iq nil nil 12 0 nil   t)
                  )
               )
            )
         )
      )
   )
)


(defun xsliceProcessMesh
   ( )

   (setq b* (getvar "splframe"))   ; dismantle mesh
   (setvar "splframe" 0)           ;    into 3D faces, lines, and points
   (command "_.explode" in)        ; surface fit polygon meshes will not be
   (setvar "splframe" b*)          ;    dismantled correctly if "splframe"=1
   (setq b  (ssget "_p" ll))       ; do not slice components on locked layers
   (if b
      (progn
         (setq
            b# (sslength b)
            j# 0
         )
         (if d+ 
            (setq
               b+ (ssadd)
               v+ 0
            )
         )
         (if d-
            (setq
               b- (ssadd)
               v- 0
            )
         )
         (while (> b# j#)
            (setq
               in (ssname b j#)
               id (entget in)
               it (cdr (assoc 0 id))
            )
            (cond
               (
                  (= "3DFACE" it)
                  (setq
                     c0 (cdr (setq i0 (assoc 10 id)))
                     c1 (cdr (setq i1 (assoc 11 id)))
                     c2 (cdr (setq i2 (assoc 12 id)))
                     c3 (cdr (setq i3 (assoc 13 id)))
                     ; corners
                     d0 (scalarProduct nv (mapcar '- c0 p1))
                     d1 (scalarProduct nv (mapcar '- c1 p1))
                     d2 (scalarProduct nv (mapcar '- c2 p1))
                     d3 (scalarProduct nv (mapcar '- c3 p1))
                     d< (max d0 d1 d2 d3)
                     d> (min d0 d1 d2 d3)
                     ; distances from slicing plane
                     ie (assoc 70 id)
                     ; visibility of edges
                  )
                  (cond
                     (
                        (and (<=    tol  d<) (>= (- tol) d>))
                        ; 3D face crosses slicing plane
                        (xsliceProcessFace)
                     )
                     (
                        (<=    tol  d<)
                        ; 3D face is entirely on positive side
                        (if d+
                           (progn (ssadd in b+) (setq v+ (+ 4 v+)))
                           (entdel in)
                        )
                     )
                     (
                        (>= (- tol) d>)
                        ; 3D face is entirely on negative side
                        (if d-
                           (progn (ssadd in b-) (setq v- (+ 4 v-)))
                           (entdel in)
                        )
                     )
                     (
                        t
                        ; 3D face is entirely on the slicing plane
                        (if d+
                           (progn (ssadd in b+) (setq v+ (+ 4 v+)))
                           (progn (ssadd in b-) (setq v- (+ 4 v-)))
                        )
                     )
                  )
               )
               (
                  (= "LINE" it)
                  (setq
                     i0 (assoc 10 id)
                     i1 (assoc 11 id)
                     ; data groups of start point and end point
                     d0 (scalarProduct nv (mapcar '- (cdr i0) p1))
                     d1 (scalarProduct nv (mapcar '- (cdr i1) p1))
                     ; distances from slicing plane
                  )
                  (cond
                     (
                        (or
                           (and (<=    tol  d0) (>= (- tol) d1))
                           (and (>= (- tol) d0) (<=    tol  d1))
                        )
                        ; line crosses slicing plane
                        (xsliceProcessLine)
                     )
                     (
                        (or  (<=    tol  d0) (<=    tol  d1))
                        ; line is entirely on positive side
                        (if d+
                           (progn (ssadd in b+) (setq v+ (+ 2 v+)))
                           (entdel in)
                        )
                     )
                     (
                        (or  (>= (- tol) d0) (>= (- tol) d1))
                        ; line is entirely on negative side
                        (if d-
                           (progn (ssadd in b-) (setq v- (+ 2 v-)))
                           (entdel in)
                        )
                     )
                     (
                        t
                        ; line is entirely on the slicing plane
                        (if d+
                           (progn (ssadd in b+) (setq v+ (+ 2 v+)))
                           (progn (ssadd in b-) (setq v- (+ 2 v-)))
                        )
                     )
                  )
               )
               (
                  t   ; (= "POINT" it)
                  (setq d0
                     (scalarProduct nv (mapcar '- (cdr (assoc 10 id)) p1))
                  )
                  (cond
                     (
                        (<=    tol  d0)
                        ; point is on positive side
                        (if d+
                           (progn (ssadd in b+) (setq v+ (1+ v+)))
                           (entdel in)
                        )
                     )
                     (
                        (>= (- tol) d0)
                        ; point is on negative side
                        (if d-
                           (progn (ssadd in b-) (setq v- (1+ v-)))
                           (entdel in)
                        )
                     )
                     (
                        t
                        ; point is on the slicing plane
                        (if d+
                           (progn (ssadd in b+) (setq v+ (1+ v+)))
                           (progn (ssadd in b-) (setq v- (1+ v-)))
                        )
                     )
                  )
               )
            )
            (setq j# (1+ j#))
         )
         (if d+
            (if (> 8192 (setq b# (sslength b+)))
               (sewProcessSet b+ b# v+)
               (princ
                  (if ger
                     (strcat
                        "Das Zusammenfassen der "
                        (itoa b#)
                        " Teile zu einem Polyflchennetz ist nicht mglich"
                        " (maximal 8191).\n"
                     )
                     (strcat
                        "Cannot reassemble "
                        (itoa b#)
                        " components to a polyface mesh"
                        " (not more than 8191).\n"
                     )
                  )
               )
            )
         )
         (if d-
            (if (> 8192 (setq b# (sslength b-)))
               (sewProcessSet b- b# v-)
               (princ
                  (if ger
                     (strcat
                        "Das Zusammenfassen der "
                        (itoa b#)
                        " Teile zu einem Polyflchennetz ist nicht mglich"
                        " (maximal 8191).\n"
                     )
                     (strcat
                        "Cannot reassemble "
                        (itoa b#)
                        " components to a polyface mesh"
                        " (not more than 8191).\n"
                     )
                  )
               )
            )
         )
      )
   )
   (setq b* nil)
)



(defun doNotSlice   ; fr Objekte, die nicht von der Ebene geschnitten werden
   ( )              ; for objects that are not intersected by the plane

   (setq n# (1+ n#))
   (if
      (cond
         (
            (= "LINE" it)
            (or
               (and (not d+) (or (<=    tol  d0) (<=    tol  d1)))
               (and (not d-) (or (>= (- tol) d0) (>= (- tol) d1)))
            )
         )
         (
            (or (= "RAY" it) (= "XLINE" it))
            (or
               (and (not d+) (<=    tol  d0))
               (and (not d-) (>= (- tol) d0))
            )
         )
         (
            (or (= "3DFACE" it) (= "POLYLINE" it))
            (or
               (and (not d+) (<=    tol  d<))
               (and (not d-) (>= (- tol) d>))
            )
         )
      )
      (progn
         (ssadd in u)    ; Objects situated entirely on the undesired side
         (setq u# (1+ u#))
         (redraw in 3)   ; get highlighted and prepared for deletion.
      )
   )
)



;;;   Unterprogramme 3. Ordnung fr xsliceProcessFace
;;;   zum Modifizieren der Elementdatenliste einer geschnittenen 3d-Flche
;;;
;;;   Die Schnittkanten sollen in jedem Fall sichtbar sein;
;;;   aber diejenigen Kanten sollen unsichtbar werden,
;;;   welche bei der eventuell ntigen Aufteilung einer entstehenden Flche
;;;   lngs der ursprnglichen Diagonalen nach dem Schnitt zu einer Auenkante
;;;   eines neuen Dreiecks oder Vierecks werden.


;;;   3rd order subroutines for xsliceProcessFace
;;;   for modifying the entity data list of a sliced 3D face
;;;
;;;   Edges generated by slicing should be visible in any case
;;;   [with the exception of the former "diagonals" that did not lie on the
;;;    slicing plane but became outer edges of new triangles or quadrangles
;;;    created because a pentangle or a hexangle had to be split].
;;;   Edges that retained their places should also retain their visibility
;;;   or invisibility.


;;  Standard-Routine
;;  Standard routine

(defun modFace
   (
      cp cq cr cs   ; neue Lage der Eckpunkte, falls nicht nil

      k0            ; Bitcode: fr Kanten, die sichtbar gemacht werden sollen,
                    ;          wird das entsprechende Bit auf Null gesetzt
      k1            ; Bitcode: fr Kanten, die unsichtbar werden sollen,
                    ;          wird das entsprechende Bit auf Eins gesetzt

      f*            ; Flag: Flche muss neu erzeugt werden
      f+            ; Flag: Flche liegt auf der positiven Seite
      /
      md            ; modifizierte Elementdatenliste
   )

   ;|
      cp cq cr cs   ; new positions of the corners if not nil

      k0            ; code: bit corresponding to an edge that has to become
                    ;       visible is set to zero
      k1            ; code: bit corresponding to an edge that has to become
                    ;       invisible is set to one

      f*            ; flag: face has to be created newly
      f+            ; flag: face is on positive side
      /
      md            ; modified entity data list
   |;

   (setq md (subst (cons 70 (logior k1 (logand k0 (cdr ie)))) ie id))
   (if cp (setq md (subst (cons 10 cp) i0 md)))
   (if cq (setq md (subst (cons 11 cq) i1 md)))
   (if cr (setq md (subst (cons 12 cr) i2 md)))
   (if cs (setq md (subst (cons 13 cs) i3 md)))
   ((if f* entmake entmod) md)
   (if b*
      (if f+
         (progn (ssadd (if f* (entlast) in) b+) (setq v+ (+ 4 v+)))
         (progn (ssadd (if f* (entlast) in) b-) (setq v- (+ 4 v-)))
      )
   )   ; b* is not nil if the face is a component of a dismantled mesh
)


;;  Modifizieren der Elementdatenliste einer 3d-Flche, der die zweite Ecke
;;  abgeschnitten wird [fr AutoCAD Release 14]
;;  [Ausgabe: viereckiger Teil des entstehenden Fnfecks bzw. Sechsecks]

;;  Modifying the data list of a 3D face whose second corner is cut
;;  [for AutoCAD release 14]
;;  [Output: quadrilateral part of a new pentangle or hexangle]

(defun modFace-1
   (
      f*   ; Flag: Flche muss neu erzeugt werden
      f+   ; Flag: Flche liegt auf der positiven Seite
      /
      md   ; modifizierte Elementdatenliste
   )

   ;|
      f*   ; flag: face has to be created newly
      f+   ; flag: face is on positive side
      /
      md   ; modified entity data list
   |;

   (setq md
      (subst
         (cons 70 (logior 8 (logand 1 (cdr ie)) (lsh (logand 2 (cdr ie))  1)))
         ie   ; new edge inserted between first and second edge
         (subst
            (cons 13 c2)
            i3
            (subst (cons 12 iq) i2 (subst (cons 11 ip) i1 id))
         )
      )
   )
   ((if f* entmake entmod) md)
   (if b*
      (if f+
         (progn (ssadd (if f* (entlast) in) b+) (setq v+ (+ 4 v+)))
         (progn (ssadd (if f* (entlast) in) b-) (setq v- (+ 4 v-)))
      )
   )   ; b* is not nil if the face is a component of a dismantled mesh
)


;;  Modifizieren der Elementdatenliste einer 3d-Flche, der die vierte Ecke
;;  abgeschnitten wird [fr AutoCAD Release 14]
;;  [Ausgabe: viereckiger Teil des entstehenden Fnfecks bzw. Sechsecks]

;;  Modifying the data list of a 3D face whose fourth corner is cut
;;  [for AutoCAD release 14]
;;  [Output: quadrilateral part of a new pentangle or hexangle]

(defun modFace-3
   (
      f*   ; Flag: Flche muss neu erzeugt werden
      f+   ; Flag: Flche liegt auf der positiven Seite
      /
      md   ; modifizierte Elementdatenliste
   )

   ;|
      f*   ; flag: face has to be created newly
      f+   ; flag: face is on positive side
      /
      md   ; modified entity data list
   |;

   (setq md
      (subst
         (cons 70 (logior 1 (logand 8 (cdr ie)) (lsh (logand 4 (cdr ie)) -1)))
         ie   ; new edge inserted between third and fourth edge
         (subst
            (cons 11 c2)
            i1
            (subst (cons 12 ir) i2 (subst (cons 13 is) i3 id))
         )
      )
   )
   ((if f* entmake entmod) md)
   (if b*
      (if f+
         (progn (ssadd (if f* (entlast) in) b+) (setq v+ (+ 4 v+)))
         (progn (ssadd (if f* (entlast) in) b-) (setq v- (+ 4 v-)))
      )
   )   ; b* is not nil if the face is a component of a dismantled mesh
)


;;  Modifizieren der Elementdatenliste einer 3d-Flche, der die erste Ecke
;;  abgeschnitten wird [fr AutoCAD Release 2000]
;;  [Ausgabe: viereckiger Teil des entstehenden Fnfecks bzw. Sechsecks]

;;  Modifying the data list of a 3D face whose first corner is cut
;;  [for AutoCAD release 2000]
;;  [Output: quadrilateral part of a new pentangle or hexangle]

(defun modFace-0
   (
      f*   ; Flag: Flche muss neu erzeugt werden
      f+   ; Flag: Flche liegt auf der positiven Seite
      /
      md   ; modifizierte Elementdatenliste
   )

   ;|
      f*   ; flag: face has to be created newly
      f+   ; flag: face is on positive side
      /
      md   ; modified entity data list
   |;

   (setq md
      (subst
         (cons 70 (logior 4 (logand 8 (cdr ie)) (lsh (logand 1 (cdr ie))  1)))
         ie   ; new edge inserted between fourth and first edge
         (subst
            (cons 12 c1)
            i2
            (subst (cons 11 iq) i1 (subst (cons 10 ip) i0 id))
         )
      )
   )
   ((if f* entmake entmod) md)
   (if b*
      (if f+
         (progn (ssadd (if f* (entlast) in) b+) (setq v+ (+ 4 v+)))
         (progn (ssadd (if f* (entlast) in) b-) (setq v- (+ 4 v-)))
      )
   )   ; b* is not nil if the face is a component of a dismantled mesh
)


;;  Modifizieren der Elementdatenliste einer 3d-Flche, der die dritte Ecke
;;  abgeschnitten wird [fr AutoCAD Release 2000]
;;  [Ausgabe: viereckiger Teil des entstehenden Fnfecks bzw. Sechsecks]

;;  Modifying the data list of a 3D face whose third corner is cut
;;  [for AutoCAD release 2000]
;;  [Output: quadrilateral part of a new pentangle or hexangle]

(defun modFace-2
   (
      f*   ; Flag: Flche muss neu erzeugt werden
      f+   ; Flag: Flche liegt auf der positiven Seite
      /
      md   ; modifizierte Elementdatenliste
   )

   ;|
      f*   ; flag: face has to be created newly
      f+   ; flag: face is on positive side
      /
      md   ; modified entity data list
   |;

   (setq md
      (subst
         (cons 70 (logior 8 (logand 4 (cdr ie)) (lsh (logand 2 (cdr ie)) -1)))
         ie   ; new edge inserted between third and fourth edge
         (subst
            (cons 10 c1)
            i0
            (subst (cons 11 ir) i1 (subst (cons 12 is) i2 id))
         )
      )
   )
   ((if f* entmake entmod) md)
   (if b*
      (if f+
         (progn (ssadd (if f* (entlast) in) b+) (setq v+ (+ 4 v+)))
         (progn (ssadd (if f* (entlast) in) b-) (setq v- (+ 4 v-)))
      )
   )   ; b* is not nil if the face is a component of a dismantled mesh
)



;____________________________________________________________________________;



(if (wcmatch (ver) "*(de)")



   ;;;   Funktion BGELN
   ;;;   bringt Objekte, welche fr eine 2d-Zeichnung vorgesehen sind,
   ;;;   in eine Ebene.
   ;;;
   ;;;   BGELN verschiebt bzw. projiziert die ausgewhlten
   ;;;   Punkte, Konstruktionslinien, Strahlen, Linien, Kreise, Bgen,
   ;;;   Ellipsen, 2d-Polylinien, 2d- und 3d-Splines in ein und dieselbe
   ;;;   Ebene, die parallel zur xy-Ebene des aktuellen BKS liegt.
   ;;;
   ;;;   Die gewnschte Hhe der Ebene [z-Koordinate im aktuellen BKS]
   ;;;   kann bestimmt werden durch das Eingeben des Zahlenwerts per Tastatur
   ;;;   oder durch das Anklicken eines Punktes,
   ;;;   der bereits die richtige Z-Koordinate besitzt.
   ;;;   Voreingestellt ist  z = 0.
   ;;;
   ;;;   Die Verschiebung bzw. Projektion erfolgt senkrecht zur xy-Ebene.
   ;;;   Die z-Koordinaten werden angepasst, jedoch smtliche x- und y-
   ;;;   Koordinaten im aktuellen BKS bleiben erhalten.
   ;;;
   ;;;   Konstruktionslinien, Strahlen und Linien werden allerdings
   ;;;   nicht verndert, wenn sie senkrecht zur xy-Ebene stehen;
   ;;;   Kreise, Bgen, Ellipsen, 2d-Polylinien und 2d-Splines werden
   ;;;   nur dann verndert, wenn sie bereits parallel zur xy-Ebene sind;
   ;;;   denn andernfalls geht BGELN davon aus,
   ;;;   dass die aktuelle Lage der Objekte den Benutzerwnschen entspricht
   ;;;   ["im Zweifel fr den Angeklagten"].
   ;;;   Deshalb bleiben auch 3d-Polylinien, 3d-Flchen, Netze, Regionen
   ;;;   und andere 3d-Objekte unangetastet.
   ;;;   [Mit den AutoCAD-Befehlen "Kreis", "Bogen", "Ellipse" und "Plinie"
   ;;;    entstehen von vornherein nur Objekte, die parallel zur xy-Ebene
   ;;;    des aktuellen BKS sind.
   ;;;    Anders beim Befehl "Spline"; ein versehentliches Erzeugen
   ;;;    von 3d-Splines ist leicht mglich.]
   ;;;
   ;;;   Objekte mit einer von Null verschiedenen Objekthhe
   ;;;   knnen nicht ausgewhlt werden.
   ;;;   Objekte auf gesperrten Layern werden generell nicht verndert.
   ;;;   Auch Schraffuren, Bemaungen und Texte sowie Objekte in Blcken
   ;;;   bleiben unbehandelt.


   (defun c:bgeln
      (
         /
         s      ; Auswahlsatz

         zz     ; gewnschte z-Koordinate im aktuellen BKS

         s#     ; Anzahl der gewhlten Objekte [auf nicht gesperrten Layern]
         l#     ; Anzahl der gewhlten Objekte auf gesperrten Layern
         m#     ; Anzahl der Elemente, die whrend der Bearbeitung
                ;        verndert wurden

         e#     ; Index des aktuell bearbeiteten Elements
         en     ; Elementname
         ed     ; Elementdatenliste
         et     ; Typ des Elements
         ep     ; Anfangspunkt, Mittelpunkt bzw. Kontrollpunkt des Elements
         er     ; Endpunkt bzw. Richtungsvektor des Elements
         ez     ; z-Koordinate des ersten Kontrollpunkts im aktuellen BKS
         e0     ; Datengruppe aus ed
         e1     ; Datengruppe aus ed
         dm     ; Flag: Elementdaten wurden modifiziert

         kn     ; Schlsselnummer der aktuell bearbeiteten Datengruppe

         ld     ; Datenliste des aktuell berprften Layers
         ll     ; Liste aller gesperrten Layer der Zeichnung

         tt     ; temporres Testflag

         ger    ; Flag: deutsche Version

         tol    ; Toleranz

         echo   ; Systemvariable "cmdecho" [command echo]
         errr   ; systemeigene Fehlerbearbeitungs-Routine
      )

      (setq ger t)
      (standardInitiate)
      (ironSelect)
      (ironInput)
      (lockedFilter)
      (ironProcess)
      (standardTerminate)
   )

)



;;;   Function IRON
;;;   puts objects onto a plane intended for a 2D drawing.
;;;
;;;   IRON moves or projects the selected points, xlines, rays, lines,
;;;   circles, arcs, ellipses,2D polylines, 2D and 3D splines
;;;   onto the same plane parallel to the current UCS XY plane.
;;;
;;;   The desired elevation of the plane [Z coordinate in current UCS]
;;;   may be defined by entering the Z value on the keyboard
;;;   or by specifying a point that already has the right Z coordinate.
;;;   Default value is  Z = 0.
;;;
;;;   The translation or projection direction is perpendicular
;;;   to the XY plane.
;;;   Z coordinates get modified;
;;;   but all X and Y coordinates in current UCS are maintained.
;;;
;;;   Xlines, rays, and lines will not be changed if they are
;;;   perpendicular to the XY plane;
;;;   circles, arcs, ellipses, 2D polylines and 2D splines
;;;   will only be changed if they are parallel to the XY plane already;
;;;   oterwise IRON considers their current positions
;;;   as being corresponding to user's wishes
;;;   ["in dubio pro reo"].
;;;   This is also the reason why 3D polylines, 3D faces, meshes,
;;;   regions, and other 3D objects are filtered out from selection.
;;;   [From the start, the "circle", "arc", "ellipse" and "pline" commands
;;;    always create objects parallel to the current UCS XY plane.
;;;    On the other hand, the "spline" command easily may create
;;;    3D splines by mistake.]
;;;
;;;   Objects with non-zero thickness are excluded from selection.
;;;   Objects on locked layers generally do not get modified.
;;;   IRON has no effect on hatches, dimensions, texts, and blocks.


(defun c:iron
   (
      /
      s      ; selection set

      zz     ; desired Z coordinate in current UCS

      s#     ; number of objects selected [on unlocked layers]
      l#     ; number of objects selected on locked layers
      m#     ; number of objects modified during processing

      e#     ; index of object currently worked on
      en     ; entity name
      ed     ; entity data list
      et     ; type of entity
      ep     ; start point, center, or vertex of entity
      er     ; end point or direction vector of entity
      ez     ; current UCS Z coordinate of first control point
      e0     ; data group from ed
      e1     ; data group from ed
      dm     ; flag: entity data were modified

      kn     ; key number of current data group

      ld     ; data list of layer currently tested
      ll     ; list of all locked layers of the drawing

      tt     ; temporary test flag

      ger    ; flag: German version

      tol    ; tolerance

      echo   ; "cmdecho" system variable [command echo]
      errr   ; system's error handling routine
   )

   (setq ger (wcmatch (ver) "*(de)"))
   (standardInitiate)
   (ironSelect)
   (ironInput)
   (lockedFilter)
   (ironProcess)
   (standardTerminate)
)



;;;   Unterprogramme 1. Ordnung fr BGELN
;;;   1st order subroutines for IRON


(defun ironSelect
   ( )

   (setq tt t)
   (while tt
      (princ
         (if ger
            (strcat
               " - Punkte, Konstruktionslinien, Strahlen, Linien,"
               " Kreise, Bgen, Ellipsen, 2d-Polylinien, Splines -"
            )
            (strcat
               " - points, xlines, rays, lines,"
               " circles, arcs, ellipses, 2D polylines, splines -"
            )
         )
      )
      (setq s
         (ssget
            '(
               (-4 . "<or")
                  (0 . "POINT")
                  (0 . "XLINE") (0 . "RAY")
                  (0 . "LINE")
                  (0 . "CIRCLE") (0 . "ARC") (0 . "ELLIPSE")
                  (0 . "LWPOLYLINE")
                  (-4 . "<and")
                     (0 . "POLYLINE")   ; no 3D polylines,
                     (-4 . "<not") (-4 . "&") (70 . 88) (-4 . "not>")
                  (-4 . "and>")         ; no polygon/polyface meshes
                  (0 . "SPLINE")
               (-4 . "or>")
               (-4 . "=") (39 . 0.0)    ; zero thickness
            )   ; Filtering by extrusion direction is not possible yet
         )      ; because relational tests concerning group 210
      )         ; do not allow a tolerance ["=" or "/=" only].
      (if s
         (setq
            s# (sslength s)
            tt nil
         )
         (princ
            (if ger
               "\nEs wurde keine gltige Auswahl getroffen."
               "\nNo valid selection made."
            )
         )
      )
   )
)



(defun ironInput
   ( )

   (initget 129)   ; arbitrary input
   (setq           ; null input [just "Enter"] returns null string [""]
      zz
         (getpoint
            (if ger
               (strcat
                  "Punkt mit gewnschter z-Koordinate "
                  "im aktuellen BKS angeben "
                  "oder Zahlenwert eingeben <0>: "
               )
               (strcat
                  "Specify a point whose Z coordinate becomes definitive "
                  "or enter desired Z coordinate in current UCS <0>: "
               )
            )
         )
      zz (if (= 'str (type zz)) (atof zz) (caddr zz))
   )   ; (atof "") returns 0.0
)



(defun ironProcess
   ( )

   (if s
      (progn
         (princ "\n")
         (setq
            m# 0
            e# 0
         )
         (while (> s# e#)
            (setq
               en (ssname s e#)
               ed (entget en)
               et (cdr (assoc 0 ed))
            )
            (cond
               (
                  (= "POINT" et)
                  (ironProcessPoint)
               )
               (
                  (or (= "XLINE" et) (= "RAY" et))
                  ; change only if direction vector is
                  ; not perpendicular to current UCS XY plane
                  (setq
                     e1 (assoc 11 ed)
                     er (trans (cdr e1) 0 1 t)
                  )
                  (if
                     (not
                        (and
                           (equal 0.0 (car  er) tol)
                           (equal 0.0 (cadr er) tol)
                        )
                     )
                     (ironProcessXline|ray)
                  )
               )
               (
                  (= "LINE" et)
                  ; change only if line is
                  ; not perpendicular to current UCS XY plane
                  (setq
                     e0 (assoc 10 ed)
                     e1 (assoc 11 ed)
                     ep (trans (cdr e0) 0 1)
                     er (trans (cdr e1) 0 1)
                  )
                  (if
                     (not
                        (and
                           (equal (car  ep) (car  er) tol)
                           (equal (cadr ep) (cadr er) tol)
                        )
                     )
                     (ironProcessLine)
                  )
               )
               (
                  (or (= "CIRCLE" et) (= "ARC" et) (= "ELLIPSE" et))
                  ; change only if parallel to current UCS XY plane
                  (setq er (trans (cdr (assoc 210 ed)) 0 1 t))
                  (if
                     (and
                        (equal 0.0 (car  er) tol)
                        (equal 0.0 (cadr er) tol)
                     )
                     (ironProcessCircle|arc|ellipse)
                  )
               )
               (
                  (= "LWPOLYLINE" et)
                  ; change only if parallel to current UCS XY plane
                  (setq er (trans (cdr (assoc 210 ed)) 0 1 t))
                  (if
                     (and
                        (equal 0.0 (car  er) tol)
                        (equal 0.0 (cadr er) tol)
                     )
                     (ironProcessLwpolyline)
                  )
               )
               (
                  (= "POLYLINE" et)
                  ; [Regarding 2D polylines, AutoCAD R14 and 2000
                  ;  use this format for curve fit polylines only.]
                  ; change only if parallel to current UCS XY plane
                  (setq er (trans (cdr (assoc 210 ed)) 0 1 t))
                  (if
                     (and
                        (equal 0.0 (car  er) tol)
                        (equal 0.0 (cadr er) tol)
                     )
                     (ironProcessPolyline)
                  )
               )
               (
                  (= "SPLINE" et)
                  (if (zerop (logand 8 (cdr (assoc 70 ed))))
                     (ironProcess3Dspline)
                     (ironProcess2Dspline)
                  )
               )
            )
            (princ
               (if ger
                  (strcat "\015"
                     (itoa m#)
                     (if (= 1 m#) " Objekt" " Objekte")
                     " verndert. "
                  )
                  (strcat "\015"
                     (itoa m#)
                     (if (= 1 m#) " object" " objects")
                     " modified. "
                  )
               )
            )
            (setq e# (1+ e#))
         )
      )
   )
)



;;;   Unterprogramme 2. Ordnung fr ironProcess
;;;   2nd order subroutines for ironProcess


(defun ironProcessPoint
   ( )

   (setq ep (trans (cdr (setq e0 (assoc 10 ed))) 0 1))
   (if (not (equal zz (caddr ep) tol))
      (progn
         (entmod   ; move point if required
            (subst (cons 10 (trans (list (car ep) (cadr ep) zz) 1 0)) e0 ed)
         )
         (setq m# (1+ m#))
      )
   )
)



(defun ironProcessXline|ray
   ( )

   (if (not (equal 0.0 (caddr er) tol))
      (setq
         ed
            (subst   ; project direction vector into XY plane if required
               (cons 11 (trans (list (car er) (cadr er) 0.0) 1 0 t))
               e1
               ed
            )
         dm t
      )
   )
   (setq ep (trans (cdr (setq e0 (assoc 10 ed))) 0 1))
   (if (not (equal zz (caddr ep) tol))
      (setq
         ed
            (subst   ; move start point or "center" if required
               (cons 10 (trans (list (car ep) (cadr ep) zz) 1 0))
               e0
               ed
            )
         dm t
      )
   )
   (if dm (progn (entmod ed) (setq m# (1+ m#) dm nil)))
)



(defun ironProcessLine
   ( )

   (if (not (equal zz (caddr ep) tol))
      (setq
         ed
            (subst   ; modify start point if required
               (cons 10 (trans (list (car ep) (cadr ep) zz) 1 0))
               e0
               ed
            )
         dm t
      )
   )
   (if (not (equal zz (caddr er) tol))
      (setq
         ed
            (subst   ; modify end point if required
               (cons 11 (trans (list (car er) (cadr er) zz) 1 0))
               e1
               ed
            )
         dm t
      )
   )
   (if dm (progn (entmod ed) (setq m# (1+ m#) dm nil)))
)



(defun ironProcessCircle|arc|ellipse
   ( )

   ; translate center point from OCS to UCS
   (setq ep (trans (cdr (setq e0 (assoc 10 ed))) en 1))
   (if (not (equal zz (caddr ep) tol))
      (progn
         (entmod
            (subst   ; move center if required
               (cons 10 (trans (list (car ep) (cadr ep) zz) 1 en))
               e0
               ed
            )
         )
         (setq m# (1+ m#))
      )
   )
)



(defun ironProcessLwpolyline
   ( )

   (setq
      e0 (assoc 38 ed)   ; elevation of object plane above WCS origin
      e1 (- zz (caddr (trans '(0.0 0.0 0.0) 0 1)))
   )
   (if (not (equal e1 (cdr e0) tol))
      (progn             ; modify if required
         (entmod (subst (cons 38 e1) e0 ed))
         (setq m# (1+ m#))
      )
   )

)



(defun ironProcessPolyline
   ( )

   (setq
      e0 (assoc 10 ed)   ; "dummy" point containing elevation
      e1 (- zz (caddr (trans '(0.0 0.0 0.0) 0 1)))
   )
   (if (not (equal e1 (cadddr e0) tol))
      (progn             ; modify if required
         (entmod (subst (list 10 0.0 0.0 e1) e0 ed))
         (setq m# (1+ m#))
      )
   )

)



(defun ironProcess3Dspline
   ( )

   (foreach kn '(10 11)              ; Scan control points and fit points ...
      (while (setq e0 (assoc kn ed))
         (setq ep (trans (cdr e0) 0 1))
         (if (equal zz (caddr ep) tol)
            (setq                    ; ... mask gradually ...
               ed (subst (cons "m" (cdr e0)) e0 ed)
            )
            (setq                    ; ... and replace if required; ...
               ed (subst
                     (cons "m" (trans (list (car ep) (cadr ep) zz) 1 0))
                     e0
                     ed
                  )
               dm t
            )
         )
      )
      (while (setq e0 (assoc "m" ed))
         (setq ed (subst (cons kn (cdr e0)) e0 ed))
      )                              ; ... remove masks.
   )
   ; In case of multiple existence of a point in the data list,
   ; a single call of the "subst" function will replace several data groups.
   ; This is why the number of loop runs should not be controlled by
   ; groups 73 and 74 (number of control points and fit points of spline).

   (foreach kn '(12 13)              ; Scan start and end tangent vector ...
      (setq e1 (assoc kn ed))
      (if e1
         (progn
            (setq er (trans (cdr e1) 0 1 t))
            (if (not (equal 0.0 (caddr er) tol))
               (setq
                  ed
                     (if             ; ... if perpendicular to XY plane ...
                        (and
                           (equal 0.0 (car  er) tol)
                           (equal 0.0 (cadr er) tol)
                        )
                        (append      ; ... then remove ...
                           (reverse (cdr (member e1 (reverse ed))))
                           (cdr (member e1 ed))
                        )
                        (subst       ; ... else replace.
                           (cons
                              kn
                              (trans (list (car er) (cadr er) 0.0) 1 0 t)
                           )
                           e1
                           ed
                        )
                     )
                  dm t
               )
            )
         )
      )
   )
   (if dm (progn (entmod ed) (setq m# (1+ m#) dm nil)))
)



(defun ironProcess2Dspline
   ( )

   (setq
      kn 10   ; scan control points first
      e0 (assoc kn ed)
      ep (trans (cdr e0) 0 1)
      ez (caddr ep)
      tt t
   )
   (while tt
      (if (if dm (not (equal ez (caddr ep) tol)) (equal zz ez tol))
         (setq
            dm nil   ; spline is not parallel to XY plane or
            tt nil   ; its elevation is correct already, so don't change
         )
         ; Test of extrusion direction [similar to polylines and circles]
         ; may deliver wrong results in case of linear splines; 
         ; that's why a more difficult method is necessary.
         (progn
            (setq
               ed (subst   ; modify and mask gradually
                     (cons "m" (trans (list (car ep) (cadr ep) zz) 1 0))
                     e0
                     ed
                  )
               dm t
               e0 (assoc kn ed)
            )
            (if e0
               (setq ep (trans (cdr e0) 0 1))
               (progn
                  (while (setq e0 (assoc "m" ed))
                     (setq ed (subst (cons kn (cdr e0)) e0 ed))
                  )   ; remove masks
                  (if (= 10 kn)
                     (progn
                        (setq      ; after control points
                           kn 11   ; scan fit points
                           e0 (assoc kn ed)
                        )
                        (if e0
                           (setq ep (trans (cdr e0) 0 1))
                           (setq tt nil)   ; there are no fit points
                        )
                     )
                     (setq tt nil)   ; all points were scanned
                  )
               )
            )
         )
      )
   )
   (if dm (progn (entmod ed) (setq m# (1+ m#) dm nil)))
)



;____________________________________________________________________________;



;;;   Funktion POLYFIT
;;;   erstellt 2d-Polylinien mit geradlinigen Segmenten
;;;   entlang des Verlaufs von gleichmig geteilten ebenen Kurven.
;;;
;;;   Linien, Kreise, Kreisbgen, Ellipsen, elliptische Bgen,
;;;   2d-Polylinien und 2d-Splines knnen ausgewhlt werden.
;;;   Deren Objekthhe muss Null betragen.
;;;
;;;   Es kann gewhlt werden, ob die Kurven
;;;   in eine bestimmte Anzahl von Abschnitten
;;;   oder in Abschnitte von bestimmter Lnge geteilt werden sollen
;;;   [vgl. AutoCAD-Befehle "Teilen" und "Messen"].
;;;
;;;   Hinweis:
;;;   Bei der vorgenommenen Aufteilung werden zwar
;;;   die Kurven-Abschnitte gleich lang, aber nicht unbedingt die daraufhin
;;;   erstellten Polylinien-Segmente;
;;;   je strker die ursprnglichen Kurven-Abschnitte gekrmmt sind,
;;;   umso strker werden die erzeugten Polylinien-Segmente verkrzt.
;;;
;;;   Wenn ein gewhltes Objekt geschlossen ist,
;;;   so erstellt POLYFIT eine geschlossene Polylinie;
;;;   andernfalls eine offene Polylinie.
;;;
;;;   Die Polylinien werden auf dem aktuellen Layer
;;;   mit den aktuellen Farb- und Linientypeigenschaften erstellt
;;;   [unabhngig von den ursprnglichen Kurven].
;;;
;;;   Durch die Umwandlung von Kurven in Polylinien
;;;   wird die Ausfhrung einiger AutoCAD-Befehle erleichtert
;;;   bzw. erst ermglicht:
;;;   - Verbinden mit anderen Objekten, Erstellen geschlossener Objekte
;;;     mittels "pedit"
;;;   - Extrudieren
;;;   - Erstellen von tabellarischen Flchen und Regelflchen
;;;   - Markieren umgrenzter Bereiche
;;;     beispielsweise fr "xclip", "clipit", "extrim", "wipeout"
;;;   Auerdem verbessert sich die Kompatibilitt beim Export
;;;   in andere CAD-Systeme [z. B. frhere AutoCAD-Versionen].


;;;   Function POLYFIT
;;;   creates 2D polylines with straight line segments
;;;   along equivalently divided length or perimeter of planar curves.
;;;
;;;   Lines, circles, arcs, ellipses, 2D polylines, and 2D splines
;;;   can be selected.
;;;   Their thickness must be equal to zero.
;;;
;;;   The user is allowed to choose whether the curves are to be divided
;;;   into a certain number of sections
;;;   or into sections of a certain length
;;;   [cf. AutoCAD "divide" and "measure" commands].
;;;
;;;   Note that a curve gets divided into sections of equal length
;;;   but the generated polyline segments differ in length;
;;;   the sharper a curve section is bending, the more a generated
;;;   polyline segment is shortened.
;;;
;;;   If a selected object is closed, POLYFIT will create
;;;   a closed polyline.
;;;
;;;   The polylines are created on the current layer
;;;   with the current color and linetype assignments
;;;   [independently of the original curves].
;;;
;;;   The POLYFIT conversion function may assist or enable
;;;   execution of various AutoCAD commands:
;;;   - joining with other objects, creating closed objects
;;;     by means of "pedit"
;;;   - extruding
;;;   - creating tabulated or ruled surfaces
;;;   - marking boundaries of areas,
;;;     especially for "xclip", "clipit", "extrim", "wipeout"
;;;   Besides, compatibility is improved concerning export
;;;   into other CAD systems [especially former AutoCAD versions].


(defun c:polyfit
   (
      /
      s      ; Auswahlsatz der zu bearbeitenden Objekte
      p      ; Auswahlsatz der Teilungspunkte

      dd     ; Lnge oder Anzahl der Objekt-Abschnitte [temporr]

             ; die folgenden Variablen bleiben
             ; fr die nchsten Funktionsaufrufe erhalten:
      ; mpolyfit ; Methode des Aufteilens
      ; lpolyfit ; Lnge der Objekt-Abschnitte
      ; npolyfit ; Anzahl der zu erstellenden Abschnitte

      s#     ; Anzahl der zu bearbeitenden Objekte
      e#     ; Index des aktuell bearbeiteten Objekts
      en     ; Elementname
      ed     ; Elementdatenliste
      et     ; Typ des Elements
      eo     ; Flag: Objekt ist offen

      vd     ; fr Linien: Richtungsvektor

      pc     ; fr Kreise, Bgen und Ellipsen: Mittelpunkt

      as     ; fr Ellipsen: Startwinkel
      ae     ;               Endwinkel
      va     ;               Vektor der groen Halbachse
      vb     ;               Vektor der kleinen Halbachse

      nn     ; fr Polylinien: Elementname des aktuell bearbeiteten
             ;                 Scheitelpunkts

      el     ; Erhebung der Objektebene gegenber dem WKS-Ursprung
      vn     ; Normalenvektor zur Objektebene

      ps     ; Startpunkt
      pe     ; Endpunkt bzw. aktuell bearbeiteter Punkt
      p#     ; Anzahl der Teilungspunkte
      i#     ; Index des aktuell bearbeiteten Teilungspunkts

      tt     ; temporres Testflag

      r14    ; Flag: Release 14
      ger    ; Flag: deutsche Version

      tol    ; Toleranz

      echo   ; Systemvariable "cmdecho" [command echo]
      errr   ; systemeigene Fehlerbearbeitungs-Routine
   )

   ;|
      /
      s      ; selection set of objects to be divided
      p      ; selection set of division points

      dd     ; length or number of object sections [temporary]

             ; the following variables are kept for future function calls:
      ; mpolyfit ; method of division
      ; lpolyfit ; length of object sections
      ; npolyfit ; number of object sections

      s#     ; number of objects to be divided
      e#     ; index of object currently worked on
      en     ; entity name
      ed     ; entity data list
      et     ; type of entity
      eo     ; flag: object is open

      vd     ; concerning lines: direction vector

      pc     ; concerning circles, arcs, ellipses: center point

      as     ; concerning ellipses: start angle
      ae     ;                      end angle
      va     ;                      vector of major half axis
      vb     ;                      vector of minor half axis

      nn     ; concerning polylines: entity name of current vertex

      el     ; elevation of object plane above WCS origin
      vn     ; normal vector of object plane

      ps     ; start point
      pe     ; end point or current point
      p#     ; number of division points
      i#     ; index of current division point

      tt     ; temporary test flag

      r14    ; flag: release 14
      ger    ; flag: German version

      tol    ; tolerance

      echo   ; "cmdecho" system variable [command echo]
      errr   ; system's error handling routine
   |;

   (setq ger (wcmatch (ver) "*(de)"))
   (standardInitiate)
   (polyfitSelect)
   (polyfitInput)
   (polyfitProcess)
   (standardTerminate)
)



;;;   Unterprogramme 1. Ordnung zu POLYFIT
;;;   1st Order Subroutines for POLYFIT


(defun polyfitSelect
   ( )

   (setq tt t)
   (while tt
      (princ
         (if ger
            (strcat
               " - Linien, Kreise, Kreisbgen, Ellipsen, elliptische Bgen,"
               " 2d-Polylinien, 2d-Splines -"
            )
            "\n - lines, circles, arcs, ellipses, 2D polylines, 2D splines -"
         )
      )
      (setq s
         (ssget
            '(
               (-4 . "<or")
                  (0 . "LINE")
                  (0 . "CIRCLE")
                  (0 . "ARC")
                  (0 . "ELLIPSE")
                  (0 . "LWPOLYLINE")
                  (-4 . "<and")
                     (0 . "POLYLINE")      ; no 3D polylines,
                     (-4 . "<not") (-4 . "&") (70 . 88) (-4 . "not>")
                  (-4 . "and>")            ; no polygon/polyface meshes
                  (-4 . "<and")
                     (0 . "SPLINE")
                     (-4 . "&") (70 . 8)   ; planar splines only
                  (-4 . "and>")
               (-4 . "or>")
               (-4 . "=") (39 . 0.0)       ; zero thickness
            )
         )
      )
      (if s
         (setq
            s# (sslength s)
            tt nil
         )
         (princ
            (if ger
               "\nEs wurde keine gltige Auswahl getroffen."
               "\nNo valid selection made."
            )
         )
      )
   )
)



(defun polyfitInput
   ( )

   (setq r14 (wcmatch (ver) "*14*"))
   (if (= "Length" mpolyfit)
      (progn
         (if (and (numberp lpolyfit) (< tol lpolyfit))
            (progn
               (setq lpolyfit (float lpolyfit))
               (initget
                  6   ; positive only
                  (if ger "Lnge Anzahl _Length Number" "Length Number")
               )
               (setq dd
                  (getreal
                     (if ger
                        (if r14
                           (strcat "\n"
                              "aufteilen nach Anzahl/Lnge <Lnge="
                              (rtos lpolyfit)
                              ">: "
                           )
                           (strcat "\n"
                              "Lnge der Kurven-Abschnitte eingeben oder "
                              "[aufteilen nach Anzahl] <Lnge="
                              (rtos lpolyfit)
                              ">: "
                           )
                        )
                        (if r14
                           (strcat "\n"
                              "divide by Number/Length <Length="
                              (rtos lpolyfit)
                              ">: "
                           )
                           (strcat "\n"
                              "Enter length of curve sections or "
                              "[divide by Number] <Length="
                              (rtos lpolyfit)
                              ">: "
                           )
                        )
                     )
                  )
               )
            )
            (progn
               (initget
                  7   ; positive only, no null [not just "Enter"]
                  (if ger "Lnge Anzahl _Length Number" "Length Number")
               )
               (setq dd
                  (getreal
                     (if ger
                        (if r14
                           "\naufteilen nach Anzahl/<Lnge>: "
                           (strcat "\n"
                              "Lnge der Kurven-Abschnitte eingeben oder "
                              "[aufteilen nach Anzahl]: "
                           )
                        )
                        (if r14
                           "\ndivide by Number/<Length>: "
                           (strcat "\n"
                              "Enter length of curve sections or "
                              "[divide by Number]: "
                           )
                        )
                     )
                  )
               )
            )
         )
         (cond
            (
               (= "Length" dd)
               (polyfitInputLength)
            )
            (
               (= "Number" dd)
               (polyfitInputNumber)
               (setq mpolyfit "Number")
            )
            (
               (not dd)
               nil                  ; default length accepted
            )
            (
               t
               (setq lpolyfit dd)   ; new length
            )
         )
      )
      (progn   ; (/= "Length" mpolyfit)
         (setq mpolyfit "Number")
         (if (not (numberp npolyfit)) (setq npolyfit 10))
         (setq npolyfit (fix npolyfit))
         (if (or (> 2 npolyfit) (< 32766 npolyfit)) (setq npolyfit 10))
         (setq tt t)
         (while tt
            (initget
                6   ; positive only
                (if ger "Lnge Anzahl _Length Number" "Length Number")
            )
            (setq dd
               (getint
                  (if ger
                     (if r14
                        (strcat "\n"
                           "aufteilen nach Lnge/Anzahl <Anzahl="
                           (itoa npolyfit)
                           ">: "
                        )
                        (strcat "\n"
                           "Anzahl der Abschnitte eingeben oder "
                           "[aufteilen nach Lnge] <Anzahl="
                           (itoa npolyfit)
                           ">: "
                        )
                     )
                     (if r14
                        (strcat "\n"
                           "divide by Length/Number <Number="
                           (itoa npolyfit)
                           ">: "
                        )
                        (strcat "\n"
                           "Enter number of sections or "
                           "[divide by Length] <Number="
                           (itoa npolyfit)
                           ">: "
                        )
                     )
                  )
               )
            )
            (cond
               (
                  (= "Length" dd)
                  (polyfitInputLength)
                  (setq
                     mpolyfit "Length"
                     tt nil
                  )
               )
               (
                  (= "Number" dd)
                  (polyfitInputNumber)
               )
               (
                  (not dd)
                  (setq tt nil)   ; default number accepted
               )
               (
                  (or (> 2 dd) (< 32766 dd))
                  (princ
                     (if ger
                        (strcat "\n"
                           "Die Anzahl muss mindestens 2 "
                           "und hchstens 32766 betragen."
                        )
                        (strcat "\n"
                           "Minimum number is 2 "
                           "and maximum number is 32766."
                        )
                     )            ; invalid number, try again
                  )
               )
               (
                  t
                  (setq
                     npolyfit dd
                     tt nil       ; new valid number
                  )
               )
            )
         )
      )
   )
)



(defun polyfitProcess
   ( )

   (setq e# 0)
   (while (> s# e#)
      (setq
         en (ssname s e#)
         ed (entget en)
         et (cdr (assoc 0 ed))
      )
      (cond
         (
            (= "LINE" et)
            (setq
               eo t                       ; open
               ps (cdr (assoc 10 ed))     ; start point in WCS
               pe (cdr (assoc 11 ed))     ; end point in WCS
               vd (mapcar '- pe ps)       ; direction vector
               vn (cdr (assoc 210 ed))    ; extrusion direction
            )
            (if (not (equal 0.0 (scalarProduct vd vn) tol))
               (setq vn (normalize (vectorProduct vd (vectorProduct vd vn))))
            )   ; set vn perpendicular to vd if required
            (setq
               el (scalarProduct ps vn)   ; elevation
               ps (trans ps 0 vn)         ; start point in OCS
               ps (list (car ps) (cadr ps))
               pe (trans pe 0 vn)         ; end point in OCS
               pe (list (car pe) (cadr pe))
            )
         )
         (
            (= "CIRCLE" et)
            (setq
               eo nil                      ; closed
               vn (cdr (assoc 210 ed))     ; extrusion direction
               pc (cdr (assoc  10 ed))     ; center in OCS
               el (caddr pc)               ; elevation
               ps
                  (polar
                     pc
                     (getvar "snapang")    ; start angle
                     (cdr (assoc 40 ed))   ; radius
                  )
               ps (list (car ps) (cadr ps))
            )
         )
         (
            (= "ARC" et)
            (setq
               eo t                        ; open
               vn (cdr (assoc 210 ed))     ; extrusion direction
               pc (cdr (assoc  10 ed))     ; center in OCS
               el (caddr pc)               ; elevation
               ps
                  (polar
                     pc
                     (cdr (assoc 50 ed))   ; start angle
                     (cdr (assoc 40 ed))   ; radius
                  )
               ps (list (car ps) (cadr ps))
               pe
                  (polar
                     pc
                     (cdr (assoc 51 ed))   ; end angle
                     (cdr (assoc 40 ed))   ; radius
                  )
               pe (list (car pe) (cadr pe))
            )
         )
         (
            (= "ELLIPSE" et)
            (setq
               vn (cdr (assoc 210 ed))    ; extrusion direction
               pc (cdr (assoc  10 ed))    ; center in WCS
               el (scalarProduct pc vn)   ; elevation
               va (cdr (assoc  11 ed))    ; major half axis in WCS
               as (cdr (assoc  41 ed))    ; start angle
               ae (cdr (assoc  42 ed))    ; end angle
            )
            (if
               (and (= 0.0 as) (= (* 2.0 pi) ae))
               (setq
                  eo nil                  ; closed
                  ps
                     (trans
                        (mapcar '+ pc va)
                        0
                        vn                ; start point in OCS
                     )
                  ps (list (car ps) (cadr ps))
               )
               (setq
                  eo t                    ; open
                  vb
                     (mapcar '(lambda (c) (* (cdr (assoc 40 ed)) c))
                        (vectorProduct vn va)
                     )                    ; minor half axis in WCS
                  ps
                     (trans
                        (mapcar '+
                           pc
                           (mapcar '(lambda (c) (* (cos as) c)) va)
                           (mapcar '(lambda (c) (* (sin as) c)) vb)
                        )
                        0
                        vn                ; start point in OCS
                     )
                  ps (list (car ps) (cadr ps))
                  pe
                     (trans
                        (mapcar '+
                           pc
                           (mapcar '(lambda (c) (* (cos ae) c)) va)
                           (mapcar '(lambda (c) (* (sin ae) c)) vb)
                        )
                        0
                        vn                ; end point in OCS
                     )
                  pe (list (car pe) (cadr pe))
               )
            )
         )
         (
            (= "LWPOLYLINE" et)
            (setq
               vn (cdr (assoc 210 ed))               ; extrusion direction
               el (cdr (assoc  38 ed))               ; elevation
               ps (cdr (assoc  10 ed))               ; start point
            )
            (if (= 1 (logand 1 (cdr (assoc 70 ed))))
               (setq eo nil)                         ; closed
               (setq
                  eo t                               ; open
                  pe (cdr (assoc 10 (reverse ed)))   ; end point
               )
            )
         )
         (
            (= "POLYLINE" et)
            (setq
               vn (cdr (assoc 210 ed))        ; extrusion direction
               el (cadddr (assoc 10 ed))      ; elevation
               nn (entnext en)
               ps (cdr (assoc 10 (entget nn)))
               ps (list (car ps) (cadr ps))   ; start point
            )
            (if (= 1 (logand 1 (cdr (assoc 70 ed))))
               (setq eo nil)                           ; closed
               (progn
                  (setq
                     eo t                              ; open
                     pe ps
                  )
                  (while
                     (=
                        "VERTEX"
                        (cdr
                           (assoc 0 (setq ed (entget (setq nn (entnext nn)))))
                        )
                     )
                     (setq
                        pe (cdr (assoc 10 ed))
                        pe (list (car pe) (cadr pe))   ; end point
                     )
                  )
               )
            )
         )
         (
            t   ; (= "SPLINE" et)
            (setq
               vn (cdr (assoc 210 ed))           ; extrusion direction
               ps (trans (cdr (assoc 10 ed)) 0 vn)
               el (caddr ps)                     ; elevation
               ps (list (car ps) (cadr ps))      ; start point
            )
            (if (= 1 (logand 1 (cdr (assoc 70 ed))))
               (setq eo nil)                     ; closed
               (setq
                  eo t                           ; open
                  pe (trans (cdr (assoc 10 (reverse ed))) 0 vn)
                  pe (list (car pe) (cadr pe))   ; end point
               )
            )
         )
      )
      (if (= "Length" mpolyfit)
         (command "_.measure" en lpolyfit)
         (command "_.divide"  en npolyfit)
      )
      (setq p  (ssget "_p"))
      (if (and p (= "POINT" (cdr (assoc 0 (entget (ssname p 0))))))
         (progn
            (setq
               p# (sslength p)
               i# p#
               ed (list (cons 210 vn))
            )
            (if       ; check whether the end point of an open object
               (and   ; is needed to complete the data list
                  eo
                  (not
                     (equal
                        pe
                        (reverse
                           (cdr
                              (reverse
                                 (trans
                                    (cdr
                                       (assoc 10 (entget (ssname p (1- p#))))
                                    )
                                    0
                                    vn
                                 )
                              )
                           )
                        )
                        tol
                     )
                  )
               )
               (setq
                  ed (cons (cons 10 pe) ed)
                  p# (1+ p#)
               )
            )
            (while (< 0 i#)   ; include all division points in data list
               (setq
                  i# (1- i#)
                  pe (trans (cdr (assoc 10 (entget (ssname p i#)))) 0 vn)
                  ed (cons (list 10 (car pe) (cadr pe)) ed)
               )
            )
            (if   ; include start point in data list if required
               (or
                  eo
                  (and
                     (= "Length" mpolyfit)
                     (or
                        (/= "POLYLINE" et)
                        (/= 5 (cdr (assoc 70 (entget en))))
                     )
                  )
               )
               (setq
                  ed (cons (cons 10 ps) ed)
                  p# (1+ p#)
               )
            )
            (entmake
               (cons
                  '(0 . "LWPOLYLINE")
                  (cons
                     '(100 . "AcDbEntity")
                     (cons
                        '(100 . "AcDbPolyline")
                        (cons
                           (cons 90 p#)
                           (cons
                              (cons 70 (if eo 0 1))
                              (cons (cons 38 el) ed)
                           )
                        )
                     )
                  )
               )
            )
            (command "_.erase" p "")
         )
      )
      (setq e# (1+ e#))
   )
   (initget (if ger "Ja Nein _Yes No" "Yes No"))
   (if
      (=
         "Yes"
         (getkword
            (if ger
               (if r14
                  "\nOriginal-Objekte lschen? Ja/<Nein>: "
                  "\nOriginal-Objekte lschen? [Ja/Nein] <Nein>: "
               )
               (if r14
                  "\nDelete source objects? Yes/<No>: "
                  "\nDelete source objects? [Yes/No] <No>: "
               )
            )
         )
      )
      (command "_.erase" s "")
   )
)



;;;   Unterprogramme 2. Ordnung fr polyfitInput
;;;   2nd order subroutines for polyfitInput


(defun polyfitInputLength
   ( )

   (if (and (numberp lpolyfit) (< tol lpolyfit))
      (progn
         (setq lpolyfit (float lpolyfit))
         (initget 6)   ; positive only
         (if
            (setq dd
               (getreal
                  (strcat "\n"
                     (if ger
                        (if r14
                           "Lnge der Kurven-Abschnitte <"
                           "Lnge der Kurven-Abschnitte eingeben <"
                        )
                        (if r14
                           "Length of curve sections <"
                           "Enter length of curve sections <"
                        )
                     )
                     (rtos lpolyfit)
                     ">: "
                  )
               )
            )
            (setq lpolyfit dd)
         )
      )
      (progn
         (initget 7)   ; positive only, no null [not just "Enter"]
         (setq lpolyfit
            (getreal
               (if ger
                  (if r14
                     "\nLnge der Kurven-Abschnitte: "
                     "\nLnge der Kurven-Abschnitte eingeben: "
                  )
                  (if r14
                     "\nLength of curve sections: "
                     "\nEnter length of curve sections: "
                  )
               )
            )
         )
      )
   )
)



(defun polyfitInputNumber
   ( )

   (if (not (numberp npolyfit)) (setq npolyfit 10))
   (setq npolyfit (fix npolyfit))
   (if (or (> 2 npolyfit) (< 32766 npolyfit)) (setq npolyfit 10))
   (setq tt t)
   (while tt
      (initget 6)   ; positive only
      (if
         (setq dd
            (getint
               (strcat "\n"
                  (if ger
                     (if r14
                        "Anzahl der Abschnitte <"
                        "Anzahl der Abschnitte eingeben <"
                     )
                     (if r14
                        "Number of sections <"
                        "Enter number of sections <"
                     )
                  )
                  (itoa npolyfit)
                  ">: "
               )
            )
         )
         (if (or (> 2 dd) (< 32766 dd))
            (princ
               (if ger
                  (strcat "\n"
                     "Die Anzahl muss mindestens 2 "
                     "und hchstens 32766 betragen."
                  )
                  (strcat "\n"
                     "Minimum number is 2 "
                     "and maximum number is 32766."
                  )
               )         ; invalid number, try again
            )
            (setq
               npolyfit dd
               tt nil    ; new valid number
            )
         )
         (setq tt nil)   ; default number accepted
      )
   )
)



;____________________________________________________________________________;



(if (wcmatch (ver) "*(de)")



   ;;;   Funktion SCHNEIDERHILFE
   ;;;   zeigt Informationen ber die SCHNEIDEREI-Funktionen
   ;;;   mit Hilfe eines Browsers an.


   (defun c:schneiderhilfe
      (
         /
         hp     ; Name und vollstndiger Pfad der HTML-Hilfe-Datei

         ger    ; Flag: deutsche Version

         echo   ; Systemvariable "cmdecho" [command echo]
         errr   ; systemeigene Fehlerbearbeitungs-Routine
      )

      (setq ger t)
      (helpInitiate)
      (tailorshelpProcess)
      (standardTerminate)
   )

)



;;;   Function TAILORSHELP
;;;   displays information about the TAILORS functions
;;;   by means of a browser.


(defun c:tailorshelp
   (
      /
      hp     ; name and path of HTML help file

      ger    ; flag: German version

      echo   ; "cmdecho" system variable
      errr   ; system's error handling routine
   )

   (setq ger (wcmatch (ver) "*(de)"))
   (helpInitiate)
   (tailorshelpProcess)
   (standardTerminate)
)



;;;   Unterprogramm 1. Ordnung fr SCHNEIDERHILFE
;;;   1st order subroutine for TAILORSHELP


(defun tailorshelpProcess
   ( )

   (if
      (setq hp
         (findfile
            (if ger
               "Tailors/Deutsch/LiesMich.html"
               "Tailors/English/ReadMe.html"
            )
         )
      )
      (command "_.browser" hp)
      (alert
         (if ger
            "Die Datei Tailors/Deutsch/LiesMich.html\nwurde nicht gefunden."
            "Help file Tailors/English/ReadMe.html\nwas not found."
         )
      )
   )
)



;____________________________________________________________________________;



;;;   Unterprogramm zum Ausfiltern von Objekten auf gesperrten Layern
;;;
;;;   Die Objektwahl und die Wahl mglichst aller bentigten Optionen
;;;   mssen dem Aufruf dieses Unterprogramms vorausgehen.
;;;   "lockedFilter" sollte erst dann ausgefhrt werden, wenn
;;;   der Befehl "'layer" nicht mehr transparent aufgerufen werden kann.


;;;   Subroutine for filtering out objects on locked layers
;;;
;;;   Object selection must precede this subroutine.
;;;   "lockedFilter" should not be called until
;;;    the "'layer" command cannot be started transparently any more.


(defun lockedFilter   ; s# has to be set on the number of objects in previous
   ( )                ; selection set before running this subroutine

   (setq ld (tblnext "layer" t))
   (if (= 4 (logand 4 (cdr (assoc 70 ld))))
      (setq ll (list (cons 8 (cdr (assoc 2 ld)))))
   )
   (while (setq ld (tblnext "layer"))
      (if (= 4 (logand 4 (cdr (assoc 70 ld))))
         (setq ll (cons (cons 8 (cdr (assoc 2 ld))) ll))
      )
   )
   (if ll
      (setq ll
         (append
            '((-4 . "<not") (-4 . "<or"))
            ll
            '((-4 . "or>") (-4 . "not>"))
         )
      )
   )
   (setq s (ssget "_p" ll))
   (if s
      (progn
         (setq
            l# s#
            s# (sslength s)
            l# (- l# s#)
         )
         (cond
            (
               (= 1 l#)
               (princ
                  (if ger
                     (strcat "\n"
                        "Eines der gewhlten Objekte "
                        "liegt auf einem gesperrten Layer."
                     )
                     (strcat "\n"
                        "One of the selected objects "
                        "lies on a locked layer."
                     )
                  )
               )
            )
            (
               (< 1 l#)
               (princ
                  (strcat "\n"
                     (itoa l#)
                     (if ger
                        " der gewhlten Objekte liegen auf gesperrten Layern."
                        " of the selected objects lie on locked layers."
                     )
                  )
               )
            )
         )
      )
      (progn
         (if (= 1 s#)
            (princ
               (if ger
                  "\nDas gewhlte Objekt liegt auf einem gesperrten Layer."
                  "\nThe selected object lies on a locked layer."
               )
            )
            (princ
               (if ger
                  "\nAlle gewhlten Objekte liegen auf gesperrten Layern."
                  "\nAll selected objects lie on locked layers."
               )
            )
         )
      )
   )
)



;____________________________________________________________________________;



;;;   Initialisieren, Terminieren und Fehlerbehandlung
;;;   Initiation, termination, and error handling


;;  Initialisierende Unterprogramme
;;  Initiating subroutines

(defun standardInitiate
   ( )

   (setq echo (getvar "cmdecho"))
   (setvar "cmdecho" 0)
   (command "_.undo" "_begin")
   (setq
      errr *error*
      *error* standardError
      tol 1.0e-012
   )
)


(defun xsliceInitiate
   ( )

   (setq echo (getvar "cmdecho"))
   (setvar "cmdecho" 0)
   (command "_.undo" "_begin")
   (setq
      errr *error*
      *error* xsliceError
      tol 1.0e-012
   )
)


(defun helpInitiate
   ( )

   (setq echo (getvar "cmdecho"))
   (setvar "cmdecho" 0)
   (command "_.undo" "_begin")
   (setq
      errr *error*
      *error* standardError
   )
)


;;  Terminierendes Unterprogramm
;;  Terminating subroutine

(defun standardTerminate
   ( )

   (setq *error* errr)
   (command "_.undo" "_end")
   (setvar  "cmdecho" echo)
   (princ)
)


;;  Unterprogramme zur Fehlerbehandlung
;;  Error handling subroutines

(defun standardError
   (message)

   (princ message)
   (setq *error* errr)
   (command "_.undo" "_end")
   (setvar  "cmdecho" echo)
   (princ)
)


(defun xsliceError
   (message)

   (princ message)
   (setq *error* errr)
   (command "_.undo" "_end")
   (setvar "cmdecho" echo)
   (command "_.regen")
   (princ)
)



;____________________________________________________________________________;



;;;   Allgemein verwendbare Unterprogramme
;;;   General-purpose subroutines


;;  Durchstopunkt der Verbindungslinie zweier 3d-Punkte durch eine Ebene
;;  [gem Strahlensatz]
;;  Die Punkte mssen verschiedene Abstnde von der Ebene haben!

;;  Intersection point of a plane traversed by a line connecting two 3D points
;;  The two points must have different distances from the plane!

(defun interPoint
   (
      p1 p2   ; Punkte                           points
      d1 d2   ; deren Abstnde von der Ebene     their distances from plane
   )

   (mapcar '+
      p1
      (mapcar '(lambda (vc) (* (/ d1 (- d1 d2)) vc))
         (mapcar '- p2 p1)
      )
   )
)


;;  Prfen, ob vier 3d-Punkte in derselben Ebene liegen
;;  [wenn ja, wird t zurckgegeben; andernfalls nil]

;;  Check whether four 3D points are situated on the same plane
;;  [if yes, t will be returned; otherwise nil]

(defun coplanar
   (
      p1    ; Punkte                       points
      p2
      p3
      p4
      /
      e1    ; Einheitsvektoren vom 4.      unit vectors from 4th
      e2    ; zum 1., 2. und 3. Punkt;     to 1st, 2nd, and 3rd point;
      e3    ; nil, falls die Punkte        nil if points
            ; identisch sind               are identical

      tol   ; Toleranz                     tolerance
   )

   (setq tol 1.0e-012)
   (if
      (and
         (setq e1 (normalize (mapcar '- p1 p4)))
         (setq e2 (normalize (mapcar '- p2 p4)))
         (setq e3 (normalize (mapcar '- p3 p4)))
      )
      (equal 0.0 (scalarProduct (vectorProduct e1 e2) e3) tol)
      t
   )
)


;;  Normieren eines 3d-Vektors
;;  Unter Beibehaltung der Richtung wird die Lnge auf 1.0 gesetzt,
;;  indem alle drei Komponenten des Vektors
;;  durch dessen ursprngliche Lnge dividiert werden.
;;  Wird der Nullvektor eingegeben, so wird nil zurckgegeben.

;;  Normalize a 3D vector
;;  Direction of vector is maintained; its length is set to 1.0
;;  by dividing all three components by original length of vector.
;;  The attempt of normalizing a zero vector returns nil.

(defun normalize
   (
      v     ; Vektor           vector
      /
      d     ; dessen Lnge     its length
      tol   ; Toleranz         tolerance
   )

   (setq
      tol 1.0e-012
      d (distance '(0.0 0.0 0.0) v)
   )
   (if (not (equal 0.0 d tol))
      (mapcar '(lambda (c) (/ c d)) v)
   )
)


;;  Skalarprodukt zweier 3d-Vektoren
;;  [ergibt Null genau dann, wenn die Vektoren orthogonal zueinander sind]

;;  Scalar product of two 3D vectors
;;  [returning zero implies and is implied by
;;   both vectors being perpendicular to one another]

(defun scalarProduct
   (v1 v2)

   (+
      (* (car   v1) (car   v2))
      (* (cadr  v1) (cadr  v2))
      (* (caddr v1) (caddr v2))
   )
)


;;  Vektorprodukt zweier 3d-Vektoren
;;  [ist stets orthogonal zu beiden Vektoren;
;;   ist Nullvektor genau dann, wenn beide Vektoren parallel sind]

;;  Vector product of two 3D vectors
;;  [is always perpendicular to both vectors;
;;   returning a zero vector implies and is implied by
;;   both vectors being parallel]

(defun vectorProduct
   (v1 v2)

   (list
      (- (* (cadr  v1) (caddr v2)) (* (caddr v1) (cadr  v2)))
      (- (* (caddr v1) (car   v2)) (* (car   v1) (caddr v2)))
      (- (* (car   v1) (cadr  v2)) (* (cadr  v1) (car   v2)))
   )
)



;____________________________________________________________________________;



(princ
   (if (wcmatch (ver) "*(de)")
      "\n Schneiderei  \251 Armin Antkowiak  Mrz 2000"
      "\n Tailors  \251 Armin Antkowiak  March 2000"
   )
)
(princ)
